<?php

namespace DealerDirect\Generic\Config;

use Dealerdirect\Generic\AbstractTestCase;

/**
 * @coversDefaultClass DealerDirect\Generic\Config\AbstractConfig
 * @covers ::<!public>
 * @covers ::__construct
 */
abstract class AbstractConfigTest extends AbstractTestCase
{
    ////////////////////////////////// FIXTURES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
    const MOCK_VALUE = 'mock-value';
    const MOCK_KEY = 'mock-key';
    const MOCK_MULTIBYTE_KEY = '💥';

    private $mockConfigArray = [
        self::MOCK_KEY => self::MOCK_VALUE,
        self::MOCK_MULTIBYTE_KEY => self::MOCK_MULTIBYTE_KEY,
    ];

    /**
     * @return string
     */
    abstract public function getSubjectName();

    /**
     * @param array|null $arguments
     * @param array|null $keys
     *
     * @return ConfigInterface
     */
    public function createSubject(array $arguments = null, array $keys = null)
    {
        $className = $this->getSubjectName();

        if ($keys === null) {
            $keys = array_keys($this->mockConfigArray);
        }

        $reflect = new \ReflectionClass($className);

        if ($reflect->isAbstract() === true) {
            $instance = $this->createObjectFromAbstractClass($className);

            $instance
                ->method('getKeys')
                ->willReturn($keys)
            ;
            $this->setNonPublicProperty($instance, 'config', $arguments);
        } else {
            $instance = $reflect->newInstanceArgs([$arguments]);
        }

        /** @noinspection PhpIncompatibleReturnTypeInspection */
        return $instance;
    }

    /** @return array */
    public function getMockConfigArray()
    {
        return $this->mockConfigArray;
    }

    /////////////////////////////////// TESTS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
    /**
     * @coversNothing
     */
    public function testConfigShouldComplainWhenInstantiatedWithoutConfigArray()
    {
        $this->markTestSkippedForAbstractSubject();

        $exceptionName = $this->getCompatibleExceptionName('\TypeError');

        $this->expectException($exceptionName);
        $this->expectExceptionMessageRegExp('/null given|none given|0 passed/');

        $this->createSubject();
    }

    /**
     * @param mixed $invalidConfig
     *
     * @coversNothing
     *
     * @dataProvider provideNonArrayValues
     */
    public function testConfigShouldComplainWhenInstantiatedWithConfigThatIsNotArray($invalidConfig)
    {
        $this->markTestSkippedForAbstractSubject();

        $exceptionName = $this->getCompatibleExceptionName('\TypeError');

        $this->expectException($exceptionName);
        $this->expectExceptionMessage('must be of the type array');

        $this->createSubject($invalidConfig);
    }

    /**
     * @covers ::get
     */
    public function testConfigShouldComplainWhenValueRetrievedBeforeItIsSet()
    {
        $config = $this->createSubject([]);

        $this->expectException(ConfigException::class);

        $config->get(self::MOCK_KEY);
    }

    /**
     * @uses \DealerDirect\Generic\Config\AbstractConfig::exist
     * @covers ::getKeys
     */
    public function testConfigShouldOnlyStoreValuesForKeysItIsFamiliarWithWhenGivenValues()
    {
        $mockConfigArray = [
            self::MOCK_KEY => self::MOCK_VALUE,
            /* Adding a key that DummyConfig is not familiar with */
            'unfamiliar-key' => 'unfamiliar-value',
        ];

        $config = $this->createSubject($mockConfigArray);

        $exist = $config->exist('unfamiliar-key');

        self::assertFalse($exist, 'key provided is not expected to exist');
    }

    /**
     * @covers ::assertExist
     * @uses \DealerDirect\Generic\Config\AbstractConfig::exist
     */
    public function testConfigShouldThrowExceptionWhenAskedToAssertExistenceOfNonExistingKey()
    {
        $config = $this->createSubject($this->getMockConfigArray());

        $this->expectException(ConfigException::class);

        $config->assertExist('non-existing-key');

    }

    /**
     * @covers ::get
     * @uses \DealerDirect\Generic\Config\AbstractConfig::exist
     */
    public function testConfigShouldReturnValueRelatedToGivenKeyWhenAskedForValueRelatedToGivenKey()
    {
        $config = $this->createSubject($this->getMockConfigArray());

        $expected = self::MOCK_VALUE;
        $actual = $config->get(self::MOCK_KEY);

        self::assertEquals($expected, $actual);
    }

    /**
     * @covers ::exist
     */
    public function testConfigShouldReturnFalseWhenAskedExistenceOfNonExistingKey()
    {
        $config = $this->createSubject($this->getMockConfigArray());

        $exists = $config->exist('non-existing-key');

        self::assertFalse($exists);
    }

    /**
     * @covers ::get
     * @uses \DealerDirect\Generic\Config\AbstractConfig::exist
     */
    public function testConfigShouldAcceptKeyWhenGivenKeyIsMultiByteString()
    {
        $config = $this->createSubject($this->getMockConfigArray());

        $expected = self::MOCK_MULTIBYTE_KEY;
        $actual = $config->get(self::MOCK_MULTIBYTE_KEY);

        self::assertEquals($expected, $actual);
    }

    /**
     * @covers ::getKeys
     * @uses \DealerDirect\Generic\Config\AbstractConfig::exist
     */
    public function testConfigShouldContainKeysWhenKeysAreDefined()
    {
        $mockConfigArray = $this->getMockConfigArray();

        $config = $this->createSubject($mockConfigArray);

        $expected = array_keys($mockConfigArray);
        $actual = $config->getKeys();

        self::assertEquals($expected,$actual);
    }

    /**
     * @uses \DealerDirect\Generic\Config\AbstractConfig::getKeys
     * @covers ::get
     */
    public function testConfigShouldOnlyExceptStringForKeysWhenKeysAreDefined()
    {
        $exceptionName = $this->getCompatibleExceptionName('\TypeError');
        $this->expectException($exceptionName);

        $config = $this->createSubject([], true);

        $config->get(AbstractConfigTest::MOCK_KEY);
    }

    /**
     * @covers ::get
     * @uses \DealerDirect\Generic\Config\AbstractConfig::exist
     */
    public function testConfigShouldCheckIfKeyIsActuallySupportedWhenAskedToRetrieveValueForKey()
    {

        $unsupported = [
            'unsupported-key' => 'unsupported-value',
        ];

        $config = $this->createSubject($unsupported);

        $this->expectException(ConfigException::class);
        $this->expectExceptionMessage('is not-supported');

        $config->get('unsupported-key');

    }

    /////////////////////////////// DATAPROVIDERS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
    public function provideNonArrayValues()
    {
        return [
            'true' => [true],
            'false' => [false],
            'null' => [null],
            'integer' => [1],
            'float' => [1.2345],
            'resource' => [fopen(__FILE__, 'rb')],
            'object' => [new \stdClass()],
        ];
    }

    ////////////////////////////// UTILITY METHODS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\
    private function markTestSkippedForAbstractSubject()
    {
        $subjectName = $this->getSubjectName();
        $reflectionClass = new \ReflectionClass($subjectName);

        if ($reflectionClass->isAbstract()) {
            $message = vsprintf('Can not test constructor of abstract class "%s".', [$subjectName]);
            $this->markTestSkipped($message);
        }
    }
}
