<?php

namespace Dealerdirect\Generic\Soap;

use Dealerdirect\Generic\Category\ResponseCode;
use nusoap_client;

class Response
{
    ////////////////////////////// CLASS PROPERTIES \\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /** @var array */
    private $data = [];
    /** @var array */
    private $errors = [];
    /** @var array */
    private /** @noinspection PropertyCanBeStaticInspection */ $response = [
        'status' => 'error',
        'code' => ResponseCode::HTTP_INTERNAL_SERVER_ERROR,
        'message' => 'Internal Server Error',
        'data' => [],
        'error' => [
            'An unknown error occurred'
        ],
    ];

    /** @var int */
    private $statusCode = ResponseCode::HTTP_INTERNAL_SERVER_ERROR;

    //////////////////////////// SETTERS AND GETTERS \\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @param nusoap_client $client
     *
     * @return \Closure
     *
     * @suppress PhanUnreferencedMethod
     */
    final public function getExceptionHandler(\nusoap_client $client)
    {
        return function (\Exception $exception) use ($client) {
            $this->buildResponse($client, $exception);

            http_response_code($this->getStatusCode());

            echo $this->createOutput();
        };
    }

    /** @return array */
    final public function getBody()
    {
        return $this->response;
    }

    /** @return int */
    final public function getStatusCode()
    {
        return $this->statusCode;
    }

    //////////////////////////////// PUBLIC API \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @param nusoap_client $client
     * @param string|array|\Exception $soapResponse
     */
    final public function buildResponse(\nusoap_client $client, $soapResponse)
    {
        $this->parse($client, $soapResponse);

        if (count($this->errors) > 0) {
            $this->statusCode = ResponseCode::HTTP_INTERNAL_SERVER_ERROR;
        }

        $this->response = [
            'code' => $this->statusCode,
            'data' => $this->data,
            'error' => $this->errors,
            'message' => ResponseCode::$statusTexts[$this->statusCode],
            'status' => $this->getStatusFromCode($this->statusCode),
        ];
    }

    /**
     * @return string
     */
    final public function createOutput()
    {
        return json_encode($this->response, JSON_PRETTY_PRINT);
    }

    ////////////////////////////// UTILITY METHODS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @param \Exception $exception
     */
    private function addErrorFromException(\Exception $exception)
    {
        $this->errors[get_class($exception)] = $exception->getMessage();
    }

    /**
     * @param int $statusCode
     *
     * @return string
     */
    private function getStatusFromCode($statusCode)
    {
        $status = 'error';

        if ($statusCode >= 400 && $statusCode <= 499) {
            $status = 'fail';
        }

        if ($statusCode >= 200 && $statusCode <= 299) {
            $status = 'success';
        }

        return $status;
    }

    /**
     * @param nusoap_client $client
     * @param string|array|\Exception $soapResponse
     */
    private function parse(nusoap_client $client, $soapResponse)
    {
        if ($soapResponse instanceof \Exception) {
            $this->addErrorFromException($soapResponse);
            //$soapResponse = HealthCheck::sanitizeException($soapResponse);
            $soapResponse = self::sanitizeException($soapResponse);
        }

        $this->setStatusCodeFromClient($client);

        $this->setDataFromSoapResponse($soapResponse);

        $this->setErrorsFromClient($client);

        $this->setErrorsFromSoapResponse($soapResponse);
    }

    /**
     * @param string|array|\Exception $soapResponse
     */
    private function setDataFromSoapResponse($soapResponse)
    {
        if ($this->isValidSoapResponse($soapResponse) === true) {
            if (is_array($soapResponse) === false) {
                $soapResponse = ['status' => $soapResponse];
            }
            $this->data = $soapResponse;
        }
    }

    /**
     * @param nusoap_client $client
     */
    private function setErrorsFromClient(nusoap_client $client)
    {
        $error = $client->getError();

        if ($error !== false) {
            $this->errors['soap-error'] = $error;
            // $this->data['soap-data'] = HealthCheck::buildErrorFromString($client->responseData);

            $this->data['soap-data'] = htmlentities($client->responseData);
        }
    }

    /**
     * @param string|array|\Exception $soapResponse
     */
    private function setErrorsFromSoapResponse($soapResponse)
    {
        if (is_array($soapResponse)) {
            if (isset($soapResponse['faultstring']) && $soapResponse['faultstring'] !== '') {
                $this->errors['soap-fault'] = $soapResponse['faultstring'];
            }

            if (isset($soapResponse['message'], $soapResponse['file'], $soapResponse['line'], $soapResponse['type'])) {
                /* Error in the SOAP status check logic*/
                $this->errors['soap-exception'] = $soapResponse['message'];
            }
        }
    }

    /**
     * @param nusoap_client $client
     */
    private function setStatusCodeFromClient(nusoap_client $client)
    {
        /** @noinspection PhpUndefinedFieldInspection */
        if (isset($client->persistentConnection->response_status_line)) {
            /** @noinspection PhpUndefinedFieldInspection */
            $statusArray = explode(' ', $client->persistentConnection->response_status_line, 3);
            $this->statusCode = (int) $statusArray[1];
        }
    }

    /**
     * @param mixed $soapResponse
     *
     * @return bool
     */
    private function isValidSoapResponse($soapResponse)
    {
        return (
            $soapResponse !== null
            && $soapResponse !== ''
            && is_object($soapResponse) === false
            && (isset($soapResponse['faultcode']) === false || $soapResponse['faultcode'] === '')
        );
    }
}

/*EOF*/
