<?php

namespace DealerDirect\Generic\Config;

/**
 * Abstract Config class
 *
 * This class is meant to be extend by projects to hold project specific configuration key/value pairs
 */
abstract class AbstractConfig implements ConfigInterface
{
    const ERROR_KEY_NOT_SUPPORTED = 'Key %s is not-supported';
    /** @var string[] */
    private $config = [];

    /** {@inheritdoc} */
    abstract public function getKeys();

    /**
     * Creates an Config object from an array of key/value pairs.
     *
     * Any values that have a key which the Config object is not familiar with
     * are not set in the config but will simply be ignored.
     *
     * @param string[] $config
     */
    public function __construct(array $config)
    {
        $this->setValues($config);
    }

    /** {@inheritdoc} */
    public function assertExist($key)
    {
        if ($this->exist($key) === false) {
            $message = vsprintf(self::ERROR_UNKNOWN_KEY, [$key, implode('", "', $this->getKeys())]);
            throw new ConfigException($message);
        }
    }

    /** {@inheritdoc} */
    public function exist($key)
    {
        return in_array($key, $this->getKeys(), true);
    }

    /** {@inheritdoc} */
    public function get($key)
    {
        if (in_array($key, $this->getKeys()) === false) {
            $message = vsprintf(Self::ERROR_KEY_NOT_SUPPORTED, [$key]);
            throw new ConfigException($message);
        } elseif (array_key_exists($key, $this->config) === false) {
            $message = vsprintf(self::ERROR_GET_BEFORE_SET, [$key]);
            throw new ConfigException($message);
        }

        return $this->config[$key];

    }

    /**
     * @param array $config
     */
    private function setValues(array $config)
    {
        array_walk($config, function ($value, $key) {
            $exist = $this->exist($key);
            if ($exist === true) {
                $this->config[$key] = $value;
            }/* Unknown values are simply ignored */
        });
    }
}

/*EOF*/
