<?php

namespace DealerDirect\Generic\Config;

use Dealerdirect\Generic\AbstractTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\CoversNothing;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\UsesClass;

/**
 * @coversDefaultClass DealerDirect\Generic\Config\AbstractConfig
 *
 * @covers ::__construct
 */
#[CoversClass(AbstractConfig::class)]
#[UsesClass(AbstractTestCase::class)]
abstract class AbstractConfigTest extends AbstractTestCase
{
    ////////////////////////////////// FIXTURES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
    public const MOCK_VALUE = 'mock-value';
    public const MOCK_KEY = 'mock-key';
    public const MOCK_MULTIBYTE_KEY = '💥';

    private $mockConfigArray = [
        self::MOCK_KEY => self::MOCK_VALUE,
        self::MOCK_MULTIBYTE_KEY => self::MOCK_MULTIBYTE_KEY,
    ];

    /**
     * @return string
     */
    abstract public function getSubjectName();

    /**
     * @SuppressWarnings(PHPMD.ElseExpression)
     *
     * @param array|null $arguments
     *
     * @return ConfigInterface
     */
    public function createSubject(array $arguments = null)
    {
        $className = $this->getSubjectName();

        $reflect = new \ReflectionClass($className);

        if ($reflect->isAbstract() === true) {
            $keys = [];

            if (is_array($arguments)) {
                $keys = array_keys($this->mockConfigArray);
            }

            $instance = new class ($keys) extends AbstractConfig {
                public function __construct(
                    protected array $keys
                ) {
                }

                public function getKeys(): array
                {
                    return $this->keys;
                }
            };

            $reflectionMethod = $reflect->getMethod('setValues');
            $reflectionMethod->setAccessible(true);
            $reflectionMethod->invoke($instance, $arguments);
        } else {
            $instance = $reflect->newInstanceArgs([$arguments]);
        }

        /** @noinspection PhpIncompatibleReturnTypeInspection */
        return $instance;
    }

    /** @return array */
    public function getMockConfigArray()
    {
        return $this->mockConfigArray;
    }

    /////////////////////////////////// TESTS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
    #[CoversNothing]
    public function testConfigShouldComplainWhenInstantiatedWithoutConfigArray()
    {

        $this->expectException('TypeError');
        $this->expectExceptionMessageMatches('/null given|none given|0 passed/');

        $this->createSubject();
    }

    #[CoversNothing]
    #[DataProvider('provideNonArrayValues')]
    public function testConfigShouldComplainWhenInstantiatedWithConfigThatIsNotArray($invalidConfig)
    {
        $this->expectException('TypeError');

        $this->createSubject($invalidConfig);
    }

    public function testConfigShouldComplainWhenValueRetrievedBeforeItIsSet()
    {
        $config = $this->createSubject([]);

        $this->expectException('DealerDirect\\Generic\\Config\\ConfigException');

        $config->get(self::MOCK_KEY);
    }

    public function testConfigShouldOnlyStoreValuesForKeysItIsFamiliarWithWhenGivenValues()
    {
        $mockConfigArray = [
            self::MOCK_KEY => self::MOCK_VALUE,
            /* Adding a key that DummyConfig is not familiar with */
            'unfamiliar-key' => 'unfamiliar-value',
        ];

        $config = $this->createSubject($mockConfigArray);

        $exist = $config->exist('unfamiliar-key');

        self::assertFalse($exist, 'key provided is not expected to exist');
    }

    public function testConfigShouldThrowExceptionWhenAskedToAssertExistenceOfNonExistingKey()
    {
        $config = $this->createSubject($this->getMockConfigArray());

        $this->expectException('DealerDirect\\Generic\\Config\\ConfigException');

        $config->assertExist('non-existing-key');
    }

    public function testConfigShouldReturnValueRelatedToGivenKeyWhenAskedForValueRelatedToGivenKey()
    {
        $config = $this->createSubject($this->getMockConfigArray());

        $expected = self::MOCK_VALUE;
        $actual = $config->get(self::MOCK_KEY);

        self::assertEquals($expected, $actual);
    }

    public function testConfigShouldReturnFalseWhenAskedExistenceOfNonExistingKey()
    {
        $config = $this->createSubject($this->getMockConfigArray());

        $exists = $config->exist('non-existing-key');

        self::assertFalse($exists);
    }

    public function testConfigShouldAcceptKeyWhenGivenKeyIsMultiByteString()
    {
        $config = $this->createSubject($this->getMockConfigArray());

        $expected = self::MOCK_MULTIBYTE_KEY;
        $actual = $config->get(self::MOCK_MULTIBYTE_KEY);

        self::assertEquals($expected, $actual);
    }

    public function testConfigShouldContainKeysWhenKeysAreDefined()
    {
        $mockConfigArray = $this->getMockConfigArray();

        $config = $this->createSubject($mockConfigArray);

        $expected = array_keys($mockConfigArray);
        $actual = $config->getKeys();

        self::assertEquals($expected, $actual);
    }

    public function testConfigShouldOnlyExceptStringForKeysWhenKeysAreDefined()
    {
        $this->expectException('DealerDirect\\Generic\\Config\\ConfigException');
        $this->expectExceptionMessage('must be of type string');

        $mockKeys = [
            0 => self::MOCK_VALUE,
        ];

        $this->createSubject($mockKeys);
    }

    public function testConfigShouldCheckIfKeyIsActuallySupportedWhenAskedToRetrieveValueForKey()
    {

        $unsupported = [
            'unsupported-key' => 'unsupported-value',
        ];

        $config = $this->createSubject($unsupported);

        $this->expectException('DealerDirect\\Generic\\Config\\ConfigException');
        $this->expectExceptionMessage('are not-supported');

        $config->get('unsupported-key');
    }

    /////////////////////////////// DATAPROVIDERS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    public static function provideNonArrayValues()
    {
        return [
            'true' => [true],
            'false' => [false],
            'null' => [null],
            'integer' => [1],
            'float' => [1.2345],
            'resource' => [fopen(__FILE__, 'rb')],
            'object' => [new \stdClass()],
        ];
    }
}
