<?php

namespace Dealerdirect\Generic\Storage;

use League\Flysystem\FileNotFoundException;

/**
 * Interface for FileSystem proxy classes to implement.
 *
 * Method declarations are taken ad verbatim from League\Flysystem\FilesystemInterface
 *
 * Please note that the "update" and "write" method SHOULD NOT be implemented.
 * The "put" function should be used instead.
 */
interface FilesystemInterface
{
    /**
     * Delete a file.
     *
     * @param string $path
     *
     * @throws FileNotFoundException
     *
     * @return bool True on success, false on failure.
     */
    public static function delete($path);

    /**
     * Get a file's mime-type.
     *
     * @param string $path The path to the file.
     *
     * @throws FileNotFoundException
     *
     * @return string|false The file mime-type or false on failure.
     */
    public static function getMimetype($path);

    /**
     * Check whether a file exists.
     *
     * @param string $path
     *
     * @return bool
     */
    public static function has($path);

    /**
     * Create a file or update if exists.
     *
     * @param string $path     The path to the file.
     * @param string $contents The file contents.
     * @param array  $config   An optional configuration array.
     *
     * @return bool True on success, false on failure.
     */
    public static function put($path, $contents, array $config = []);

    /**
     * Create a file or update if exists.
     *
     * @param string   $path     The path to the file.
     * @param resource $resource The file handle.
     * @param array    $config   An optional configuration array.
     *
     * @throws \InvalidArgumentException Thrown if $resource is not a resource.
     *
     * @return bool True on success, false on failure.
     */
    public static function putStream($path, $resource, array $config = []);

    /**
     * Read a file.
     *
     * @param string $path The path to the file.
     *
     * @throws FileNotFoundException
     *
     * @return string|false The file contents or false on failure.
     */
    public static function read($path);

    /**
     * Retrieves a read-stream for a path.
     *
     * @param string $path The path to the file.
     *
     * @throws FileNotFoundException
     *
     * @return resource|false The path resource or false on failure.
     */
    public static function readStream($path);
}
