<?php

namespace Dealerdirect\Generic\VehicleInformation;

use DealerDirect\Generic\Category\Fuel;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Category\VehicleTypeCarBodyType;
use DealerDirect\Generic\Category\VehicleTypeMotorBodyType;
use DealerDirect\Generic\Repository\MakesAndModels;
use Doctrine\DBAL\Connection;
use Exception;
use ForceUTF8\Encoding;
use GuzzleHttp\Client;
use Psr\Log\LoggerInterface;

class Vwe
{
    private const VWE_LICENSE_PLATE_DATA = 'SB-RDWA-MIL-AUD';
    private const VWE_TRIM_DATA = 'SB-ATL-MMT';
    private const VWE_WOK_CODE = 'a35';
    private const VWEKEYS = [
        'ATD:MODEL' => 'model_atd',
        'ATD:TYPEID' => 'ATD_ID',
        'MIL:AANTALVERSNELLINGEN' => 'transmissie_aant',
        'MIL:CO2UITSTOOTGECOMBINEERD' => 'co2uitstoot',
        'MIL:TYPEVERSNELLINGSBAK' => 'transmissie',
        'RDW:AANTALCILINDERS' => 'cilinders',
        'RDW:AANTALDEUREN' => 'aantal_deuren',
        'RDW:AANTALZITPLAATSEN' => 'zitplaatsen',
        'RDW:BPM' => 'bpm_bedrag',
        'RDW:BRANDSTOF1' => 'brandstof1',
        'RDW:BRANDSTOF2' => 'brandstof2',
        'RDW:BRANDSTOF3' => 'brandstof3',
        'RDW:CATALOGUSPRIJS' => 'cat_prijs',
        'RDW:CILINDERINHOUD' => 'cyl_inhoud',
        'RDW:DATUMAANSPRAKELIJKHEID' => 'datum_laatste_tenaamstelling',
        'RDW:DATUMEERSTETOELATINGINTERNATIONAAL' => 'eerste_toel_dat',
        'RDW:DATUMEERSTETOELATINGNATIONAAL' => 'eerste_ins_dat',
        'RDW:DATUMVERVALAPK' => 'apkvervaldatum',
        'RDW:EMISSIECODE' => 'emissiecode',
        'RDW:HANDELSBENAMING' => 'model',
        'RDW:INRICHTING' => 'carrosserie',
        'RDW:ISTAXI' => 'is_taxi',
        'RDW:KENTEKEN' => 'kenteken',
        'RDW:KENTEKENSIGNAAL' => 'wacht_op_keuring',
        'RDW:KLEUR1' => 'kleur1',
        'RDW:MASSALEEGVOERTUIG' => 'gewicht',
        'RDW:MERK' => 'merk',
        'RDW:TENAAMSTELLENMOGELIJK' => 'tenaamstellenmogelijk',
        'RDW:VERMOGENKW' => 'vermogen',
        'RDW:VOERTUIGCLASSIFICATIE' => 'classificatie',
        'RDW:VOERTUIGSOORT' => 'voertuigsoort',
    ];
    private const REMOVEFROMMODEL = [
        'cabrio',
        'station',
        'grand tour',
        '1.9d turbo commercial',
        'king cab',
        'sportbreak',
        'sport estate',
        'sport sedan',
        'coupe',
        'aerodeck',
        'targa',
        'touring',
        '(993)',
        '(996)',
        '(997)',
        '(991)',
        'estate',
        'break',
        'vario',
        'avant',
        'roadster',
        'combi',
        'SRT-6',
    ];

    private Client $client;

    public function __construct(
        private string $url,
        private string $user,
        private string $password,
        private LoggerInterface $logger,
    ) {
        //setup Guzzle client
        $this->client = new Client([
            // You can set any number of default request options.
            'timeout' => 3,
        ]);
    }

    public function getKentekenDataVWE(string $kenteken, Connection $dbal)
    {
        $val = null;

        try {
            $val = $this->vweCreateRequest(self::VWE_LICENSE_PLATE_DATA, $kenteken);
        } catch (\GuzzleHttp\Exception\GuzzleException $e) {
            throw new VweException('VWE request failed', 999, $e);
        }

        $aanvraag = $this->vweResponseOK($val);
        if (true !== $aanvraag) {
            // Early return, no need to continue
            return $this->vweErrorIdToMsg($aanvraag);
        }

        $vehicleType = VehicleType::CAR;
        $kenteken_info = [];
        $vehicleTypeTag = '';

        foreach ($val as $key) {
            if (!array_key_exists($key['tag'], self::VWEKEYS)) {
                continue;
            }

            // Set default value
            $key['value'] ??= '';

            //VOERTUIG SOORT
            if ('RDW:VOERTUIGSOORT' == $key['tag']) {
                $vehicleTypeTag = $key['value'];
                $vehicleType = match ($key['value']) {
                    'M', // Motorfiets
                    'Q', // Quad
                    'DMV', // Driewielig motorvoertuig
                    'DMC', // Driewielig motor carier
                    => VehicleType::MOTOR,

                    'BF', // Bromfiets
                    'BM', // Brommobiel
                    'SF', // Snorfiets
                    => VehicleType::SCOOTER,

                    'CAV', // Caravan
                    'CAM', // Camper
                    => VehicleType::RECREATION,

                    // 'A', // Aanhangwagen
                    // 'B', // Bedrijfswagen
                    // 'O', // Oplegger
                    // 'P', // Personenauto
                    // 'V', // Vrachtauto
                    // 'OVG', // Overig
                    default => VehicleType::CAR,
                };
            } elseif ('MIL:TYPEVERSNELLINGSBAK' == $key['tag']) {
                $key['value'] = match (trim($key['value'])) {
                    'Automatisch geschakelde transmissie',
                    'Continu Variabele Transmissie'
                    => 2,

                    // 'Handmatig geschakelde transmissie',
                    default => 1,
                };
            }

            $datetimeKeys = [
                'RDW:DATUMEERSTETOELATINGINTERNATIONAAL', 'RDW:DATUMEERSTETOELATINGNATIONAAL', 'RDW:DATUMVERVALAPK',
            ];
            if (in_array($key['tag'], $datetimeKeys)) {
                $key['value'] = str_replace('-', '', $key['value']);
            }

            $kenteken_info[self::VWEKEYS[$key['tag']]] = utf8_encode($key['value']);
        }

        if (empty($kenteken_info['merk'])) {
            throw new VweException("Vwe didn't have any information");
        }

        $dataMapping = [
            'RDW:TENAAMSTELLENMOGELIJK' => fn ($value) => 'true' === $value,
            'RDW:KENTEKENSIGNAAL' => fn ($value) => self::VWE_WOK_CODE === $value,
        ];

        foreach ($dataMapping as $key => $callback) {
            $vweKey = self::VWEKEYS[$key];
            $kenteken_info[$vweKey] = isset($kenteken_info[$vweKey])
                ? $callback($kenteken_info[$vweKey])
                : null;
        }

        $admissionYear = substr($kenteken_info['eerste_toel_dat'] ?? '', 0, 4);
        $kenteken_info['bouwjaar'] = $admissionYear;

        if (
            isset($kenteken_info['model_atd'])
            && mb_strlen($kenteken_info['model_atd']) < mb_strlen($kenteken_info['model'])
        ) {
            $kenteken_info['model'] = $kenteken_info['model_atd'];
        }

        if (!empty($kenteken_info['model'])) {
            if (VehicleType::RECREATION == $vehicleType || VehicleType::CAR == $vehicleType) {
                try {
                    $kenteken_info['model'] = Encoding::fixUTF8($kenteken_info['model']);
                    $kenteken_info['model'] = trim(
                        strtoupper(
                            str_replace(
                                self::REMOVEFROMMODEL,
                                '',
                                strtolower(
                                    iconv('UTF-8', 'ASCII//TRANSLIT', $kenteken_info['model'])
                                )
                            )
                        )
                    );
                } catch (Exception $exception) {
                    $this->logger->error($exception->getMessage(), [
                        'license' => $kenteken, 'value' => $kenteken_info['model'],
                    ]);
                }
            }

            $kenteken_info['model'] = match ($kenteken_info['model']) {
                'SCA(C)NIC', 'SC~A(c)NIC', 'SCA?A(C)NIC' => 'SCENIC',
                'MA(C)GANE', 'M~A(c)GANE' => 'MEGANE',
                'MA(C)GANE SCA(C)NIC', 'M~A(c)GANE SC~A(c)NIC' => 'MEGANE SCENIC',
                default => $kenteken_info['model'],
            };
        }

        $makesModels = new MakesAndModels($dbal, $vehicleType);
        $kenteken_info['vehicleType'] = $vehicleType;
        $kenteken_info['tag'] = $vehicleTypeTag;
        $kenteken_info['merk'] = $makesModels->normalizeMakeName(
            $kenteken_info['merk']
        );
        $kenteken_info['makeId'] = $makesModels->getMakeIdByName(
            $kenteken_info['merk']
        );

        if (!empty($kenteken_info['model'])) {
            $kenteken_info['model'] = $makesModels->normalizeModelName(
                $kenteken_info['makeId'],
                $kenteken_info['model']
            );
            $kenteken_info['modelId'] = $makesModels->getModelIdByName(
                $kenteken_info['makeId'],
                $kenteken_info['model']
            );
        } else {
            $kenteken_info['model'] = 'OVERIGE';
            $kenteken_info['modelId'] = 0;
        }

        // Derive and set fuel type from VWE info
        $kenteken_info['fuelType'] = $this->getFuelTypeForVWEInfo(
            $kenteken_info['brandstof1'],
            $kenteken_info['brandstof2'],
            $kenteken_info['brandstof3']
        );

        $kenteken_info['bodyType'] = $this->getBodyTypeId($kenteken_info['voertuigsoort']);

        return $kenteken_info;
    }

    private function getBodyTypeId(string $bodyType): int
    {
        return match ($bodyType) {
            'A' => VehicleTypeCarBodyType::TRAILER, // Aanhangwagen
            'B' => VehicleTypeCarBodyType::UNKNOWN, // Bedrijfswagen
            'M' => VehicleTypeMotorBodyType::UNKNOWN, // Motorfiets
            'O' => VehicleTypeCarBodyType::TRAILER, // Oplegger
            'P' => VehicleTypeCarBodyType::UNKNOWN, // Personenauto
            'V' => VehicleTypeCarBodyType::TRUCK, // Vrachtauto
            'CAV' => VehicleTypeCarBodyType::CARAVAN, // Caravan
            'CAM' => VehicleTypeCarBodyType::CAMPER, // Camper
            'BF' => VehicleTypeMotorBodyType::MOPED_45, // Bromfiets
            'BM' => VehicleTypeMotorBodyType::MICROCAR, // Brommobiel
            'SF' => VehicleTypeMotorBodyType::MOPED_25, // Snorfiets
            'Q' => VehicleTypeMotorBodyType::QUAD, // Quad
            'DMV' => VehicleTypeMotorBodyType::TRIKE, // Driewielig motor voertuig
            'DMC' => VehicleTypeMotorBodyType::TRIKE, // Driewielig motor carier
            'OVG' => VehicleTypeCarBodyType::UNKNOWN, // Overig
            'T' => VehicleTypeMotorBodyType::TRIKE, // Trike
            'X' => VehicleTypeCarBodyType::UNKNOWN, // Taxi
            // Aanhangwagens en verwisselbare getrokken machines voor land- en bosbouw
            'ALB' => VehicleTypeCarBodyType::UNKNOWN,
            'LBT' => VehicleTypeCarBodyType::UNKNOWN, // Land- of bosbouwtrekker
            'MM' => VehicleTypeCarBodyType::UNKNOWN, // Mobiele machines
            'VBS' => VehicleTypeCarBodyType::UNKNOWN, // Voertuig met beperkte snelheid
            default => VehicleTypeCarBodyType::UNKNOWN,
        };
    }

    public function getTrimByLicensePlate($licensePlate)
    {
        $trimValues = [
            'trim' => null,
            'trim_extended' => null,
        ];

        try {
            $request = $this->vweCreateRequest(self::VWE_TRIM_DATA, $licensePlate);
        } catch (\GuzzleHttp\Exception\GuzzleException $e) {
            throw new VweException('VWE request failed', 999, $e);
        }

        $responseOK = $this->vweResponseOK($request);
        if (true === $responseOK) {
            $requiredKeys = [
                'ATL:UITVOERING' => 'trim',
                'ATL:UITVOERING_LANG' => 'trim_extended',
            ];

            foreach ($request as $value) {
                if (array_key_exists($value['tag'], $requiredKeys)) {
                    $trimValues[$requiredKeys[$value['tag']]] = $value['value'];
                }
            }
        } else {
            $msg = $this->vweErrorIdToMsg($responseOK);

            throw new VweException($msg, $responseOK);
        }

        return $trimValues;
    }

    /**
     * Returns the logically derived DD fuel type value,
     * based on given VWE fuel string values
     *
     * @param string $fuelType1 Primary fuel type
     * @param string $fuelType2 Optional second fuel type
     * @return int
     */
    private function getFuelTypeForVWEInfo(...$allFuel)
    {
        $allFuel = array_map('strtolower', $allFuel);

        if (
            in_array('elektriciteit', $allFuel)
            && in_array('benzine', $allFuel)
            && in_array('lpg', $allFuel)
        ) {
            return Fuel::HYBRID_GASOLINE_LPG;
        }

        if (
            in_array('elektriciteit', $allFuel)
            && in_array('benzine', $allFuel)
        ) {
            return Fuel::HYBRID_GASOLINE;
        }

        if (
            in_array('elektriciteit', $allFuel)
            && in_array('diesel', $allFuel)
        ) {
            return Fuel::HYBRID_DIESEL;
        }

        foreach ($allFuel as $fuelType) {
            switch ($fuelType) {
                case 'benzine':
                    return Fuel::GASOLINE;
                case 'diesel':
                    return Fuel::DIESEL;
                case 'lpg':
                    return Fuel::LPG;
                case 'cng':
                    return Fuel::CNG;
                case 'waterstof':
                    return Fuel::HYDROGEN;
                case 'elektriciteit':
                    return Fuel::ELECTRICITY;
            }
        }

        return Fuel::UNKNOWN;
    }

    // VWE helper functions
    private function vweCreateRequest($messageType, $licensePlate)
    {
        $xmlName = $this->user;
        $xmlPassword = $this->password;

        $xmlMessage = <<<XML
            <bericht>
                <authenticatie>
                    <naam>{$xmlName}</naam>
                    <wachtwoord>{$xmlPassword}</wachtwoord>
                    <berichtsoort>{$messageType}</berichtsoort>
                    <referentie>test</referentie>
                </authenticatie>
                <parameters>
                    <kenteken>{$licensePlate}</kenteken>
                </parameters>
            </bericht>
            XML;

        $response = $this->client->get(sprintf(
            '%s%s',
            $this->url,
            urlencode($xmlMessage)
        ));
        if (
            200 !== $response->getStatusCode()
            // TODO add content type check
        ) {
            throw new \Exception('Vwe Request did not return a xml response body');
        }

        $xmlBody = $response->getBody()->getContents();
        // The VWE response body is 'tricky' and needs double xml decoding
        $xml = simplexml_load_string($xmlBody);
        $par = xml_parser_create();
        xml_parse_into_struct($par, $xml, $val, $ind);

        return $val;
    }

    private function vweResponseOK($xml)
    {
        foreach ($xml as &$obj) {
            if (isset($obj['tag']) && 'CODE' == $obj['tag']) {
                if (isset($obj['value']) && '00' == $obj['value']) {
                    return true;
                }

                return $obj['value'];
            }
        }
    }

    private function vweErrorIdToMsg(int $code)
    {
        return match ($code) {
            10 => 'Formaat bericht is niet correct.',
            30 => 'Geen rubrieken bij dataaccount gevonden.',
            80 => 'Geen voertuiggegevens gevonden ahv input.',
            90 => 'Naam wachtwoord is onjuist.',
            91 => 'Onbekend berichtsoort.',
            92 => 'Benodigde XML tag ontbreekt.',
            93 => 'Ongeldige of ontbrekende namespace gedefinieerd.',
            94 => 'Bericht niet actief.',
            95 => 'Saldo niet toereikend.',
            99 => 'Applicatie of onbekende fout opgetreden.',
            default => sprintf('Unknown error code $d', $code),
        };
    }
}
