<?php

namespace Dealerdirect\Generic\VehicleInformation;

use DealerDirect\Generic\Category\Fuel;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Repository\MakesAndModels;
use Doctrine\DBAL\Connection;
use GuzzleHttp\Client;

class Vwe
{
    private const VWE_LICENSE_PLATE_DATA = 'SB-RDWA-MIL-AUD';
    private const VWE_TRIM_DATA = 'SB-ATL-MMT';

    private string $url;
    private string $user;
    private string $password;
    private Client $client;

    private const VWEKEYS = [
        'RDW:KENTEKEN' => 'kenteken',
        'RDW:MERK' => 'merk',
        'RDW:HANDELSBENAMING' => 'model',
        'RDW:VOERTUIGSOORT' => 'voertuigsoort',
        'RDW:BRANDSTOF1' => 'brandstof1',
        'RDW:BRANDSTOF2' => 'brandstof2',
        'RDW:BRANDSTOF3' => 'brandstof3',
        'RDW:KLEUR1' => 'kleur1',
        'RDW:AANTALZITPLAATSEN' => 'zitplaatsen',
        'RDW:DATUMEERSTETOELATINGINTERNATIONAAL' => 'eerste_toel_dat',
        'RDW:DATUMEERSTETOELATINGNATIONAAL' => 'eerste_ins_dat',
        'RDW:DATUMAANSPRAKELIJKHEID' => 'datum_laatste_tenaamstelling',
        'RDW:DATUMVERVALAPK' => 'apkvervaldatum',
        'RDW:AANTALCILINDERS' => 'cilinders',
        'RDW:CILINDERINHOUD' => 'cyl_inhoud',
        'RDW:MASSALEEGVOERTUIG' => 'gewicht',
        'RDW:AANTALDEUREN' => 'aantal_deuren',
        'RDW:INRICHTING' => 'carrosserie',
        'RDW:VOERTUIGCLASSIFICATIE' => 'classificatie',
        'RDW:VERMOGENKW' => 'vermogen',
        'RDW:EMISSIECODE' => 'emissiecode',
        'RDW:BPM' => 'bpm_bedrag',
        'RDW:CATALOGUSPRIJS' => 'cat_prijs',
        'RDW:ISTAXI' => 'is_taxi',
        'MIL:CO2UITSTOOTGECOMBINEERD' => 'co2uitstoot',
        'MIL:TYPEVERSNELLINGSBAK' => 'transmissie',
        'MIL:AANTALVERSNELLINGEN' => 'transmissie_aant',
        'ATD:TYPEID' => 'ATD_ID',
        'ATD:MODEL' => 'model_atd'
    ];

    private const REMOVEFROMMODEL = [
        'cabrio',
        'station',
        'grand tour',
        '1.9d turbo commercial',
        'king cab',
        'sportbreak',
        'sport estate',
        'sport sedan',
        'coupe',
        'aerodeck',
        'targa',
        'touring',
        '(993)',
        '(996)',
        '(997)',
        '(991)',
        'estate',
        'break',
        'vario',
        'avant',
        'roadster',
        'combi',
        'SRT-6'
    ];

    public function __construct($url, $user, $password)
    {
        $this->url = $url;
        $this->user = $user;
        $this->password = $password;

        //setup Guzzle client
        $this->client = new Client([
            // You can set any number of default request options.
            'timeout'  => 3,
        ]);
    }

    public function getKentekenDataVWE(string $kenteken, Connection $dbal)
    {
        $val = null;
        try {
            $val = $this->vweCreateRequest(self::VWE_LICENSE_PLATE_DATA, $kenteken);
        } catch (\GuzzleHttp\Exception\GuzzleException $e) {
            throw new VweException("VWE request failed", 999, $e);
        }

        //KIJKEN OF AANVRAAG CORRECT IS
        $aanvraag = $this->vweResponseOK($val);
        if ($aanvraag === true) {
            $vehicleType = VehicleType::CAR;
            $kenteken_info = array();
            $vehicleTypeTag = '';

            foreach ($val as $key) {
                if (array_key_exists($key['tag'], self::VWEKEYS)) {
                    if (!isset($key['value'])) {
                        $key['value'] = '';
                    }

                    //VOERTUIG SOORT
                    if ($key['tag'] == 'RDW:VOERTUIGSOORT') {
                        $vehicleTypeTag = $key['value'];
                        switch ($key['value']) {
                            case 'A':
                                $vehicleType = VehicleType::CAR;
                                break; //Aanhangwagen
                            case 'B':
                                $vehicleType = VehicleType::CAR;
                                break; //Bedrijfswagen
                            case 'M':
                                $vehicleType = VehicleType::MOTOR;
                                break; //Motorfiets
                            case 'O':
                                $vehicleType = VehicleType::CAR;
                                break; //Oplegger
                            case 'P':
                                $vehicleType = VehicleType::CAR;
                                break; //Personenauto
                            case 'V':
                                $vehicleType = VehicleType::CAR;
                                break; //Vrachtauto
                            case 'CAV':
                                $vehicleType = VehicleType::RECREATION;
                                break; //Caravan
                            case 'CAM':
                                $vehicleType = VehicleType::RECREATION;
                                break; //Camper
                            case 'BF':
                                $vehicleType = VehicleType::SCOOTER;
                                break; //Bromfiets
                            case 'BM':
                                $vehicleType = VehicleType::SCOOTER;
                                break; //Brommobiel
                            case 'SF':
                                $vehicleType = VehicleType::SCOOTER;
                                break; //Snorfiets
                            case 'Q':
                                $vehicleType = VehicleType::MOTOR;
                                break; //Quad
                            case 'DMV':
                                $vehicleType = VehicleType::MOTOR;
                                break; //Driewielig motor voertuig
                            case 'DMC':
                                $vehicleType = VehicleType::MOTOR;
                                break; //Driewielig motor carier
                            case 'OVG':
                                $vehicleType = VehicleType::CAR;
                                break; //Overig
                        }
                    } elseif ($key['tag'] == 'MIL:TYPEVERSNELLINGSBAK') {
                        //TRANSMISSIE
                        switch (trim($key['value'])) {
                            case 'Handmatig geschakelde transmissie':
                                $key['value'] = '1';
                                break;
                            case 'Automatisch geschakelde transmissie':
                                $key['value'] = '2';
                                break;
                            case 'Continu Variabele Transmissie':
                                $key['value'] = '2';
                                break;
                            default:
                                $key['value'] = '1';
                        }
                    } elseif ($key['tag'] == 'RDW:DATUMEERSTETOELATINGINTERNATIONAAL') {
                        //DATUM EERSTE TOELATING AANPASSEN
                        $key['value'] = str_replace('-', '', $key['value']);
                    } elseif ($key['tag'] == 'RDW:DATUMEERSTETOELATINGNATIONAAL') {
                        //DATUM EERSTE INSCHRIJVING NL
                        $key['value'] = str_replace('-', '', $key['value']);
                    } elseif ($key['tag'] == 'RDW:DATUMVERVALAPK') {
                        //DATUM APK
                        $key['value'] = str_replace('-', '', $key['value']);
                    }

                    $kenteken_info[self::VWEKEYS[$key['tag']]] = utf8_encode($key['value']);
                }
            }

            if (empty($kenteken_info['merk'])) {
                throw new VweException("Vwe didn't have any information", 0);
            }

            $admissionYear = substr($kenteken_info['eerste_toel_dat'] ?? '', 0, 4);
            $kenteken_info['bouwjaar'] = $admissionYear;

            if (
                isset($kenteken_info['model_atd'])
                && mb_strlen($kenteken_info['model_atd']) < mb_strlen($kenteken_info['model'])
            ) {
                $kenteken_info['model'] = $kenteken_info['model_atd'];
            }



            if (!empty($kenteken_info['model'])) {
                if (
                    $vehicleType == VehicleType::RECREATION
                    || $vehicleType == VehicleType::CAR
                ) {
                    $kenteken_info['model'] = trim(strtoupper(str_replace(
                        self::REMOVEFROMMODEL,
                        '',
                        strtolower(
                            iconv(
                                'UTF-8',
                                'ASCII//TRANSLIT',
                                $kenteken_info['model']
                            )
                        )
                    )));
                }

                if (
                    $kenteken_info['model'] === 'SCA(C)NIC'
                    || $kenteken_info['model'] === 'SC~A(c)NIC'
                ) {
                    $kenteken_info['model'] = 'SCENIC';
                } elseif (
                    $kenteken_info['model'] === 'MA(C)GANE'
                    || $kenteken_info['model'] === 'M~A(c)GANE'
                ) {
                    $kenteken_info['model'] = 'MEGANE';
                } elseif (
                    $kenteken_info['model'] === 'MA(C)GANE SCA(C)NIC'
                    || $kenteken_info['model'] === 'M~A(c)GANE SC~A(c)NIC'
                ) {
                    $kenteken_info['model'] = 'MEGANE SCENIC';
                }
            }

            $makesModels = new MakesAndModels($dbal, $vehicleType);
            $kenteken_info['vehicleType'] = $vehicleType;
            $kenteken_info['tag'] = $vehicleTypeTag;
            $kenteken_info['merk'] = $makesModels->normalizeMakeName(
                $kenteken_info['merk']
            );
            $kenteken_info['makeId'] = $makesModels->getMakeIdByName(
                $kenteken_info['merk']
            );

            if (!empty($kenteken_info['model'])) {
                $kenteken_info['model'] = $makesModels->normalizeModelName(
                    $kenteken_info['makeId'],
                    $kenteken_info['model']
                );
                $kenteken_info['modelId'] = $makesModels->getModelIdByName(
                    $kenteken_info['makeId'],
                    $kenteken_info['model']
                );
            } else {
                $kenteken_info['model'] = 'OVERIGE';
                $kenteken_info['modelId'] = 0;
            }

            // Derive and set fuel type from VWE info
            $kenteken_info['fuelType'] = $this->getFuelTypeForVWEInfo(
                $kenteken_info['brandstof1'],
                $kenteken_info['brandstof2'],
                $kenteken_info['brandstof3']
            );

            return $kenteken_info;
        } else {
            $err = $this->vweErrorIdToMsg($aanvraag);
            return $err;
        }
    }

    public function getTrimByLicensePlate($licensePlate)
    {
        $trimValues = [
            'trim' => null,
            'trim_extended' => null
        ];

        try {
            $request = $this->vweCreateRequest(self::VWE_TRIM_DATA, $licensePlate);
        } catch (\GuzzleHttp\Exception\GuzzleException $e) {
            throw new VweException("VWE request failed", 999, $e);
        }

        $responseOK = $this->vweResponseOK($request);
        if ($responseOK === true) {
            $requiredKeys = [
                'ATL:UITVOERING' => 'trim',
                'ATL:UITVOERING_LANG' => 'trim_extended',
            ];

            foreach ($request as $value) {
                if (array_key_exists($value['tag'], $requiredKeys)) {
                    $trimValues[$requiredKeys[$value['tag']]] = $value['value'];
                }
            }
        } else {
            $msg = $this->vweErrorIdToMsg($responseOK);
            throw new VweException($msg, $responseOK);
        }
        return $trimValues;
    }


    /**
     * Returns the logically derived DD fuel type value,
     * based on given VWE fuel string values
     *
     * @param string $fuelType1 Primary fuel type
     * @param string $fuelType2 Optional second fuel type
     * @return int
     */
    private function getFuelTypeForVWEInfo(...$allFuel)
    {
        $allFuel = array_map('strtolower', $allFuel);

        if (
            in_array('elektriciteit', $allFuel)
            && in_array('benzine', $allFuel)
            && in_array('lpg', $allFuel)
        ) {
            return Fuel::HYBRID_GASOLINE_LPG;
        }

        if (
            in_array('elektriciteit', $allFuel)
            && in_array('benzine', $allFuel)
        ) {
            return Fuel::HYBRID_GASOLINE;
        }

        if (
            in_array('elektriciteit', $allFuel)
            && in_array('diesel', $allFuel)
        ) {
            return Fuel::HYBRID_DIESEL;
        }

        foreach ($allFuel as $fuelType) {
            switch ($fuelType) {
                case 'benzine':
                    return Fuel::GASOLINE;
                case 'diesel':
                    return Fuel::DIESEL;
                case 'lpg':
                    return Fuel::LPG;
                case 'cng':
                    return Fuel::CNG;
                case 'waterstof':
                    return Fuel::HYDROGEN;
                case 'elektriciteit':
                    return Fuel::ELECTRICITY;
            }
        }

        return Fuel::UNKNOWN;
    }

    // VWE helper functions
    private function vweCreateRequest($messageType, $licensePlate)
    {
        $xmlName = $this->user;
        $xmlPassword = $this->password;

        $xmlMessage = <<<XML
            <bericht>
                <authenticatie>
                    <naam>{$xmlName}</naam>
                    <wachtwoord>{$xmlPassword}</wachtwoord>
                    <berichtsoort>{$messageType}</berichtsoort>
                    <referentie>test</referentie>
                </authenticatie>
                <parameters>
                    <kenteken>{$licensePlate}</kenteken>
                </parameters>
            </bericht>
            XML;

        $response = $this->client->get(sprintf(
            '%s%s',
            $this->url,
            urlencode($xmlMessage)
        ));
        if (
            $response->getStatusCode() !== 200
            // TODO add content type check
        ) {
            throw new \Exception("Vwe Request did not return a xml response body");
        }

        $xmlBody = $response->getBody()->getContents();
        // The VWE response body is 'tricky' and needs double xml decoding
        $xml = simplexml_load_string($xmlBody);
        $par = xml_parser_create();
        xml_parse_into_struct($par, $xml, $val, $ind);

        return $val;
    }

    private function vweResponseOK($xml)
    {
        foreach ($xml as &$obj) {
            if (isset($obj['tag']) && $obj['tag'] == 'CODE') {
                if (isset($obj['value']) && $obj['value'] == '00') {
                    return true;
                } else {
                    return $obj['value'];
                }
            }
        }
    }

    private function vweErrorIdToMsg(int $code)
    {
        switch ($code) {
            case '10':
                $err = 'Formaat bericht is niet correct.';
                break;
            case '30':
                $err = 'Geen rubrieken bij dataaccount gevonden.';
                break;
            case '80':
                $err = 'Geen voertuiggegevens gevonden ahv input.';
                break;
            case '90':
                $err = 'Naam wachtwoord is onjuist.';
                break;
            case '91':
                $err = 'Onbekend berichtsoort.';
                break;
            case '92':
                $err = 'Benodigde XML tag ontbreekt.';
                break;
            case '93':
                $err = 'Ongeldige of ontbrekende namespace gedefinieerd.';
                break;
            case '94':
                $err = 'Bericht niet actief.';
                break;
            case '95':
                $err = 'Saldo niet toereikend.';
                break;
            case '99':
                $err = 'Applicatie of onbekende fout opgetreden.';
                break;
            default:
                $err = sprintf('Unknown error code $d', $code);
        }
        return $err;
    }
}
