<?php

namespace DealerDirect\Generic\Repository;

use Dealerdirect\Generic\AbstractTestCase;
use DealerDirect\Generic\Category\Color;
use DealerDirect\Generic\Category\Country;
use DealerDirect\Generic\Category\Locale as LocaleCategory;
use DealerDirect\Generic\Category\TranslationTypes;
use DealerDirect\Generic\Repository\CommonTranslator;
use DealerDirect\Generic\Repository\Locale;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\UsesClass;

#[CoversClass(CommonTranslator::class)]
#[UsesClass(Locale::class)]
class CommonTranslatorTest extends AbstractTestCase
{
    public function testCommonTranslatorInstanceShouldHaveDefaultLocaleDutch()
    {
        $translator = new CommonTranslator(new Locale('fr', 'FR'));
        self::assertEquals(LocaleCategory::NL_NL, (string)$translator->getDefaultLocale());
    }

    public function testMinimumTranslationsExist()
    {
        $expectedTranslations = [
            'nl_nl', 'nl_be', 'de_de', 'fr_be', 'en_us'
        ];

        $translator = new CommonTranslator(new Locale('nl', 'NL'));
        $translations = $translator->getInstalledTranslations();
        foreach ($expectedTranslations as $expectedTranslation) {
            self::assertContains($expectedTranslation, $translations);
        }
    }

    #[DataProvider('provideSupportedAlternativeLocales')]
    public function testForInvalidLocaleCorrectDefaultTranslationsShouldBeLoaded(
        Locale $locale,
        $defaultLocale,
        $matchValues
    ) {
        $translator = new CommonTranslator($locale, $defaultLocale);
        $translation = $translator->getAll(TranslationTypes::GENERAL_VALUES);

        // Assert match on several specific values, and test all methods
        self::assertTrue($translator->isTranslated('color', Color::GREEN));
        self::assertEquals($matchValues['color:green'], $translator->translate('color', Color::GREEN));
        self::assertEquals($matchValues['color:green'], $translation['color'][Color::GREEN]);

        self::assertTrue($translator->isTranslated('country', Country::NETHERLANDS));
        self::assertEquals(
            $matchValues['country:netherlands'],
            $translator->translate('country', Country::NETHERLANDS)
        );
        self::assertEquals($matchValues['country:netherlands'], $translation['country'][Country::NETHERLANDS]);
    }

    #[DataProvider('provideInstalledTranslations')]
    public function testAllTranslationsAreComplete($installedTranslation)
    {
        // TODO Only GENERAL_VALUES are tested now,
        // because OPTIONS and VEHICLE_VALUES are deeper level and or specifically for [Vehicle]Translator
        $translationTypes = [
            TranslationTypes::GENERAL_VALUES,
        ];

        foreach ($translationTypes as $translationType) {
            // Check General Values
            $baseTranslator = new CommonTranslator(new Locale('nl', 'NL'));
            $refTranslation = $baseTranslator->getAll($translationType);

            $locale = Locale::fromString($installedTranslation);
            $testTranslator = new CommonTranslator($locale);
            $checkTranslations = $testTranslator->getAll($translationType);

            foreach ($refTranslation as $category => $subset) {
                foreach ($subset as $key => $value) {
                    $isKeyTranslated = $testTranslator->isTranslated($category, $key);
                    if (!$isKeyTranslated) {
                        // Print info for debugging if assert fails
                        echo "\nAssert failed for translation '$category: $key'";
                    }
                    self::assertTrue($isKeyTranslated);
                }
            }
        }
    }

    /**
     * @return array[]
     */
    public static function provideSupportedLocales()
    {
        return [
            'LOCALE_DUTCH' => [new Locale('nl', 'NL')],
            'LOCALE_BELGIAN_DUTCH' => [new Locale('nl', 'BE')],
            'LOCALE_GERMAN' => [new Locale('de', 'DE')],
            'LOCALE_BELGIAN_FRENCH' => [new Locale('fr', 'BE')],
        ];
    }

    /**
     * @return array[]
     */
    public static function provideSupportedAlternativeLocales()
    {
        return [
            'LOCALE_FRENCH' => [
                new Locale('fr', 'FR'),
                null,
                [
                    'color:green' => 'Vert',
                    'country:netherlands' => 'Pays-Bas',
                ]
            ],
            'LOCALE_AUSTRIAN' => [
                new Locale('de', 'AT'),
                null,
                [
                    'color:green' => 'Grün',
                    'country:netherlands' => 'Niederlande',
                ]
            ],
            // Polish Locale not supported by DD, should default to English (provided as parameter)
            'LOCALE_POLISH' => [
                new Locale('pl', 'PL'),
                new Locale('en', 'US'),
                [
                    'color:green' => 'Green',
                    'country:netherlands' => 'The Netherlands',
                ]
            ],
            // 'Rare' Locale not supported by DD, should default to English (provided as parameter)
            'LOCALE_GHANEAN_HAUSA' => [
                new Locale('ha', 'GH'), // Hausa (Ghana)
                new Locale('en', 'US'),
                [
                    'color:green' => 'Green',
                    'country:netherlands' => 'The Netherlands',
                ]
            ],
        ];
    }

    /** return array[] */
    public static function provideInstalledTranslations()
    {
        $translations = CommonTranslator::getInstalledTranslations();
        $data = [];
        array_walk($translations, function ($value) use (&$data) {
            if ($value !== 'nl_nl') {
                $data['(TRANSLATION_' . $value . ')'] = [$value];
            }
        });
        ksort($data);

        return $data;
    }
}
