<?php
namespace DealerDirect\Generic\Repository;

use DealerDirect\Generic\Category\TranslationTypes;

/**
 * Class Translator
 * @package DealerDirect\Generic\Repository
 *
 * Basic Translations class, mapping array values for given keys.
 *
 * Usage example:
 *      //Set Locale, can either be Locale instance or string value:
 *      $locale = new Locale(Language::NL, Territory::NL);
 *      $localeStr = 'nl_nl';
 *
 *      $translator = new CommonTranslator($locale);
 *      $translations = $translator->getAll(TranslationTypes::GENERAL_VALUES);
 *      $translatedValue = $translator->translate('color', 5); // Green color value
 *      echo $translatedValue; // "Groen"
 *
 */
class CommonTranslator implements RepositoryInterface
{
    /** @var Locale */
    protected $locale;

    /** @var Locale */
    protected $defaultLocale;

    /** @var array Loaded translations type */
    protected $translationsType;

    /** @var array Loaded translations  */
    protected $translations = [];

    /** @var string */
    protected $defaultValue;

    /**
     * CommonTranslator constructor.
     *
     * A default "fallback" locale can be passed in case no translation files are available.
     *
     * @param Locale $locale A Locale object instance
     * @param Locale $defaultLocale Locale to use in case no translations exist for given Locale. NL is default if not passed.
     *
     */
    public function __construct(Locale $locale, Locale $defaultLocale = null)
    {
        $this->locale = $locale;
        if ($defaultLocale !== null) {
            $this->defaultLocale = $defaultLocale;
        } else {
            $this->defaultLocale = new Locale('nl', 'NL');
        }
    }

    /**
     * @param $translationsType
     * @return array
     */
    public function getAll($translationsType)
    {
        TranslationTypes::assertExists($translationsType);
        return $this->getTranslations($translationsType);
    }

    /**
     * @return Locale
     */
    public function getDefaultLocale()
    {
        return $this->defaultLocale;
    }

    /**
     * @return array
     */
    public static function getInstalledTranslations()
    {
        $translationsDir = sprintf("%s/../Lang/", __DIR__);
        return preg_grep('/^([^.])/', scandir($translationsDir));
    }

    /**
     * @param $translationsType
     * @param array|string $ids
     * @param bool $returnWithIds
     *
     * @return array
     *
     * @throws \Exception
     * @throws \InvalidArgumentException
     */
    public function get($translationsType, $ids = [], $returnWithIds = false)
    {
        TranslationTypes::assertExists($translationsType);
        $translations = $this->getTranslations($translationsType);

        $results = [];

        foreach ($ids as $id) {
            $value = $this->defaultValue;

            if (array_key_exists($id, $translations)) {
                $value = $translations[$id];
            }

            $results[$id] = $value;
        }

        if ($returnWithIds === false) {
            $results = array_values($results);
        }

        return $results;
    }

    /**
     * Returns array of translations loaded for given translations type.
     *
     * @param $translationsType
     * @param $translationsKey Key for specific translations group, within translations type
     *
     * @see TranslationTypes
     *
     * @return array
     * @throws \Exception
     */
    public function getTranslations($translationsType, $translationsKey = null)
    {
        $this->translationsType = $translationsType;
        $this->translations = $this->loadTranslations($translationsType);

        if (array_key_exists('default', $this->translations)) {
            $this->defaultValue = $this->translations['default'];
        }
        if ($translationsKey !== null) {
            $translationsKey = strtolower($translationsKey);
            if (!is_array($this->translations) || !array_key_exists($translationsKey, $this->translations)) {
                unset($this->translations);
                $errorMsg = sprintf("Error, could not find translation key '%s' in translations", $translationsKey);
                throw new \Exception($errorMsg);
            }
            $this->translations = $this->translations[$translationsKey];
            return $this->translations;
        }
        return $this->translations;
    }

    /**
     * @param string $translationsType
     * @throws \Exception
     *
     * @return array
     */
    protected function loadTranslations($translationsType)
    {
        $localeStr = $this->getValidLocale($this->locale);
        $langFile = sprintf("%s/../Lang/%s/%s", __DIR__, $localeStr, basename($translationsType));

        if (!file_exists($langFile)) {
            $errorMsg = sprintf('Error, could not load translations file. Given path: "%s"', $langFile);
            throw new \Exception($errorMsg);
        }
        return include $langFile;
    }

    /**
     * Returns given locale, or in case for given locale are no translations installed,
     * the most appropriate translations, based on Locale language.
     */
    protected function getValidLocale(Locale $locale)
    {
        $localeStr =  strtolower((string)$locale);
        $baseDir = sprintf("%s/../Lang/", __DIR__);
        $langDir = $baseDir . $localeStr;

        if (file_exists($langDir)) {
            return $localeStr;
        }
        $lang = strtolower($locale->getLanguage());

        // Find best alternative, based on language
        foreach (self::getInstalledTranslations() as $dir) {
            if (strpos($dir, $lang) === 0) {
                // first found, return as valid translations locale
                return $dir;
            }
        }

        return strtolower((string)$this->defaultLocale);
    }

    /**
     * @param string $key Key for translations group
     * @param string $subKey
     *
     * @return string The translated value (or a translation error value, specifying a missing translation)
     */
    public function translate($key, $subKey = null)
    {
        if (array_key_exists($key, $this->translations)) {
            if ($subKey === null) {
                return $this->translations[$key];
            } elseif ($subKey !== null && is_array($this->translations[$key]) && array_key_exists($subKey, $this->translations[$key])) {
                return $this->translations[$key][$subKey];
            } else {
                return sprintf('No translation for %s : %s.%s', $this->translationsType, $key, $subKey);
            }
        }
        return sprintf('No translation for %s : %s', $this->translationsType, $key);
    }

    /**
     * Check whether specified translation key (with sub key) exists
     *
     * @param $key
     * @param $subKey
     *
     * @return bool
     */
    public function isTranslated($key, $subKey = null)
    {
        if ($subKey !== null) {
            return array_key_exists($key, $this->translations)
                && is_array($this->translations[$key])
                && array_key_exists($subKey, $this->translations[$key]);
        }
        return array_key_exists($key, $this->translations);
    }
}
