<?php

namespace DealerDirect\Generic\Repository;

use Dealerdirect\Generic\AbstractTestCase;
use DealerDirect\Generic\Category\Country;
use DealerDirect\Generic\Category\CountryTerritoryCode;
use DealerDirect\Generic\Category\Language;
use DealerDirect\Generic\Category\LanguageId;
use DealerDirect\Generic\Category\Locale as LocaleCategory;
use DealerDirect\Generic\Repository\LocaleRepository;

/**
 * @coversDefaultClass DealerDirect\Generic\Repository\LocaleRepository
 */
class LocaleRepositoryTest extends AbstractTestCase
{
    private $expectedLanguages = [
        '1' => Language::NL,
        '2' => Language::NL,
        '3' => Language::DE,
        '4' => LocaleCategory::FR, //Language::FR,
        '5' => LocaleCategory::EN //Language::EN,
    ];

    /**
     * @var array expectedCountries, stored as array for fast lookup
     * extracted from database [dealer01_ddmain.countries].
     */
    private $expectedCountries = [
        '1' => 'NL',
        '2' => 'BE',
        '3' => 'DE',
        '4' => 'FR',
        '5' => 'SE',
        '6' => 'DK',
        '7' => 'NO',
        '8' => 'PL',
        '9' => 'AT',
        '10' => 'BG',
        '11' => 'CY',
        '12' => 'HR',
        '13' => 'CZ',
        '14' => 'EE',
        '15' => 'FI',
        '16' => 'GR',
        '17' => 'HU',
        '18' => 'IE',
        '19' => 'IT',
        '20' => 'LV',
        '21' => 'LT',
        '22' => 'LU',
        '23' => 'PT',
        '24' => 'RO',
        '25' => 'SK',
        '26' => 'SI', // @FIXME Note! This is incorrectly 'SL' in the database
        '27' => 'ES',
        '31' => 'CH',
    ];

    /**
     * @param $countryId
     *
     * @covers ::getCountryCodeForCountryId
     *
     * @dataProvider provideCountryIds
     */
    public function testGivenCountryIdShouldReturnCorrectCountryCode($countryId)
    {
        $expectedCountryCode = $this->expectedCountries[(string)$countryId];
        $localeRepository = new LocaleRepository();
        $countryCode = $localeRepository->getCountryCodeForCountryId($countryId);
        self::assertEquals($expectedCountryCode, $countryCode);
    }

    /**
     * @param $countryId An invalid (not used by DealerDirect) Country ID
     *
     * @covers ::getCountryCodeForCountryId
     *
     * @dataProvider provideInvalidCountryIds
     */
    public function testGivenInvalidCountryIdShouldReturnDefaultCountryCode($countryId)
    {
        $expectedCountryCode = CountryTerritoryCode::NETHERLANDS;
        $localeRepository = new LocaleRepository();
        $countryCode = $localeRepository->getCountryCodeForCountryId($countryId);
        self::assertEquals($expectedCountryCode, $countryCode);
    }

    /**
     * @param $languageId
     *
     * @covers ::getLanguageForId
     *
     * @dataProvider provideLanguageIds
     */
    public function testGivenLanguageIdShouldReturnCorrectLanguageCode($languageId)
    {
        $expectedLanguage = $this->expectedLanguages[(string)$languageId];
        $localeRepository = new LocaleRepository();
        $language = $localeRepository->getLanguageForId($languageId);
        self::assertEquals($expectedLanguage, $language);
    }

    /**
     * @param $languageId An invalid (not used by DealerDirect) Language ID
     *
     * @covers ::getLanguageForId
     *
     * @dataProvider provideInvalidLanguageIds
     */
    public function testGivenInvalidLanguageIdShouldReturnDefaultLanguage($languageId)
    {
        $expectedLanguage = Language::NL;
        $localeRepository = new LocaleRepository();
        $language = $localeRepository->getLanguageForId($languageId);
        self::assertEquals($expectedLanguage, $language);
    }

    /**
     * @param $languageId
     * @param $countryId
     * @param $expectedLocale
     *
     * @covers ::getLocaleForLanguageIdAndCountryId
     * @covers ::getLanguageForId
     * @covers ::getCountryCodeForCountryId
     * @uses \DealerDirect\Generic\Repository\Locale
     *
     * @dataProvider provideLanguageIdAndCountryIds
     */
    public function testGivenLanguageIdAndCountryIdShouldReturnCorrectLocale($languageId, $countryId, $expectedLocale)
    {
        $localeRepository = new LocaleRepository();
        $locale = $localeRepository->getLocaleForLanguageIdAndCountryId($languageId, $countryId);

        self::assertEquals($expectedLocale, (string)$locale);
    }

    /**
     * @covers ::getLocaleForLanguageIdAndCountryId
     * @covers ::getLanguageForId
     * @covers ::getCountryCodeForCountryId
     * @uses \DealerDirect\Generic\Repository\Locale
     */
    public function testUnsupportedLanguageIdAndCountryIdShouldReturnDefaultLocale()
    {
        $defaultLocale = new Locale('nl', 'NL');
        $localeRepository = new LocaleRepository();
        $locale = $localeRepository->getLocaleForLanguageIdAndCountryId(40, 33, $defaultLocale);

        self::assertEquals((string)$defaultLocale, (string)$locale);
    }

    /**
     * @return array[]
     */
    public function provideCountryIds()
    {
        return $this->getCategoriesAsDataProvider('DealerDirect\Generic\Category\Country', 'COUNTRY_CODE_');
    }

    /**
     * @return array[]
     */
    public function provideInvalidCountryIds()
    {
        return [[0], [29], ['30'], ['99']];
    }

    /**
     * @return array[]
     */
    public function provideLanguageIds()
    {
        return $this->getCategoriesAsDataProvider('DealerDirect\Generic\Category\LanguageId', 'LANG_');
    }

    /**
     * @return array[]
     */
    public function provideInvalidLanguageIds()
    {
        return [['0'], ['6'], ['10'], ['12']];
    }

    /**
     * @return array[] Returns 3 dim-arrays with values: {LanguageId, CountryId, ExpectedLocale}
     */
    public function provideLanguageIdAndCountryIds()
    {
        return [
            [LanguageId::DUTCH, Country::NETHERLANDS, LocaleCategory::NL_NL],
            [LanguageId::DUTCH, Country::BELGIUM, LocaleCategory::NL_BE],
            [LanguageId::FLEMISH, Country::BELGIUM, LocaleCategory::NL_BE],

        ];
    }

    /**
     * @return array[]
     */
    public function provideLocales()
    {
        return $this->getCategoriesAsDataProvider('DealerDirect\Generic\Category\Locale', 'LOCALE_');
    }
}
