<?php
namespace DealerDirect\Generic\Repository;

use DealerDirect\Generic\Category\VehicleType;
use PDO;

class MakesAndModels implements RepositoryInterface
{
    const CAR_STRUCTURES = '1,2,5';
    const MOTOR_STRUCTURES = '3,4';
    const RECREATION_STRUCTURES = '6';
    const SCOOTER_STRUCTURES = '7';
    const ALL_STRUCTURES = '1,2,3,4,5,6,7';

    const STATUS_ACTIVE = 1;

    /** @var PDO */
    protected $pdo;
    /** @var string */
    protected $vehicleType;

    /**
     * MakesAndModels constructor.
     *
     * @param PDO $pdo
     * @param string $vehicleType
     */
    public function __construct(PDO $pdo, $vehicleType)
    {
        VehicleType::assertExists($vehicleType);
        $this->vehicleType = $vehicleType;
        $this->pdo = $pdo;
    }

    /**
     * @return array|null
     */
    public function getAll()
    {
        $results = null;
        $pdo = $this->pdo;
        $query = $this->getAllMakesModelsQuery();
        $preparedStatement = $pdo->prepare($query);
        $queryResult = $preparedStatement->execute([':status' => self::STATUS_ACTIVE]);
        if ($queryResult !== false) {
            $unsortedResults = $preparedStatement->fetchAll(PDO::FETCH_OBJ);
            $results = $this->formatResults($unsortedResults);
        }

        return $results;
    }

    /**
     * @param string $makeName
     *
     * @return int
     */
    public function getMakeIdByName($makeName)
    {
        $result = 0;
        $makeName = $this->normalizeMakeName($makeName);
        $pdo = $this->pdo;
        $query = <<<SQL
SELECT
  merkID
FROM dealer01_webservices.SB_MERKEN
WHERE merkNaam = :makeName
SQL;
        $preparedStatement = $pdo->prepare($query);
        if ($preparedStatement !== false) {
            $params = [':makeName' => $makeName];
            $queryResult = $preparedStatement->execute($params);
            if ($queryResult !== false) {
                $row = $preparedStatement->fetchObject();
                $result = (int) $row->merkID;
            }
        }

        return $result;
    }

    /**
     * @param int $makeId
     * @param string $modelName
     *
     * @return int
     */
    public function getModelIdByName($makeId, $modelName)
    {
        $result = 0;
        $modelName = $this->normalizeModelName($makeId, $modelName);
        $pdo = $this->pdo;
        $query = <<<SQL
SELECT
  modelID
FROM dealer01_webservices.SB_MODELLEN
WHERE modelNaam = :modelName
AND merkID = :makeId
SQL;
        $preparedStatement = $pdo->prepare($query);
        if ($preparedStatement !== false) {
            $params = [':modelName' => $modelName, ':makeId' => $makeId];
            $queryResult = $preparedStatement->execute($params);
            if ($queryResult !== false) {
                $row = $preparedStatement->fetchObject();
                $result = (int) $row->modelID;
            }
        }

        return $result;
    }

    /**
     * @param bool $returnArray
     *
     * @return array|string
     */
    protected function getVehicleTypeStructure($returnArray = false)
    {
        $vehicleType = $this->vehicleType;
        switch ($vehicleType) {
            case VehicleType::CAR:
                $result = self::CAR_STRUCTURES;
                break;
            case VehicleType::SCOOTER:
                $result = self::SCOOTER_STRUCTURES;
                break;
            case VehicleType::MOTOR:
                $result = self::MOTOR_STRUCTURES;
                break;
            case VehicleType::RECREATION:
                $result = self::RECREATION_STRUCTURES;
                break;
            case VehicleType::GENERIC:
                $result = self::ALL_STRUCTURES;
                break;
            default:
                $errorMsg = sprintf(
                    'Invalid vehicleType, given vehicleType: "%s", expected one of these vehicleTypes "%s".',
                    $vehicleType,
                    implode(', ', VehicleType::getAll())
                );
                throw new \InvalidArgumentException($errorMsg);
                break;
        }
        if ($returnArray === true) {
            $result = explode(',', $result);
        }

        return $result;
    }

    /**
     * @param array $unsortedResult
     *
     * @return array
     */
    private function formatResults($unsortedResult)
    {
        $models = [];

        foreach ($unsortedResult as $row) {
            if (!array_key_exists($row->makeId, $models)) {
                $models[$row->makeId] = [];
                $models[$row->makeId]['makeId'] = (int) $row->makeId;
                $models[$row->makeId]['makeName'] = utf8_encode($row->makeName);
                $models[$row->makeId]['models'] = [];
            }
            $model = [
                'modelId' => (int) $row->modelId,
                'modelName' => utf8_encode($row->modelName),
            ];
            array_push($models[$row->makeId]['models'], $model);
        }

        return $models;
    }

    /**
     * @return string
     */
    private function getAllMakesModelsQuery()
    {
        return <<<SQL
SELECT
  mk.merkID makeId,
  md.modelID modelId,
  IF (rme.naar IS NULL OR rme.van = 'ALPINA', UPPER(mk.merkNaam), UPPER(rme.naar)) makeName,
  IF (rmo.naar IS NULL, UPPER(md.modelNaam), UPPER(rmo.naar)) modelName
FROM dealer01_webservices.SB_MERKEN mk
INNER JOIN dealer01_webservices.SB_MODELLEN md
  ON md.merkId = mk.merkId
LEFT JOIN dealer01_webservices.SB_STRUCTUUR_MODELLEN sm
  ON sm.modelID = md.modelID
LEFT JOIN dealer01_webservices.SB_REGELS_MODELLEN rmo
  ON md.merkID = rmo.merkID
  AND md.modelNaam = rmo.van
LEFT JOIN dealer01_webservices.SB_REGELS_MERKEN rme
  ON mk.merkNaam = rme.van
WHERE (
  sm.structuurID IN ({$this->getVehicleTypeStructure()})
  AND sm.active = :status
)
GROUP BY makeName, modelName
ORDER BY makeName ASC, modelName ASC
SQL;
    }

    /**
     * @param string $makeName
     *
     * @return string
     */
    public function normalizeMakeName($makeName)
    {
        $result = $makeName;
        $pdo = $this->pdo;
        $query = <<<SQL
SELECT naar
FROM dealer01_webservices.SB_REGELS_MERKEN
WHERE van = :makeName
SQL;
        $preparedStatement = $pdo->prepare($query);
        if ($preparedStatement !== false) {
            $params = [':makeName' => $makeName];
            $queryResult = $preparedStatement->execute($params);
            if ($queryResult !== false) {
                $row = $preparedStatement->fetchObject();
                if ($row !== false) {
                    $result = $row->naar;
                }
            }
        }

        return $result;
    }

    /**
     * @param int $makeId
     * @param string $modelName
     *
     * @return string
     */
    public function normalizeModelName($makeId, $modelName)
    {
        $result = $modelName;
        $pdo = $this->pdo;
        $query = <<<SQL
SELECT naar
FROM dealer01_webservices.SB_REGELS_MODELLEN
WHERE van = :modelName
AND merkID = :makeId
SQL;
        $preparedStatement = $pdo->prepare($query);
        if ($preparedStatement !== false) {
            $queryResult = $preparedStatement->execute([
                ':modelName' => $modelName,
                ':makeId' => $makeId,
            ]);
            if ($queryResult !== false) {
                $row = $preparedStatement->fetchObject();
                if ($row !== false) {
                    $result = $row->naar;
                }
            }
        }

        return $result;
    }
}
