<?php

namespace DealerDirect\Generic\Repository;

use DealerDirect\Generic\Category\Country;
use DealerDirect\Generic\Category\CountryTerritoryCode;
use DealerDirect\Generic\Category\Language;
use DealerDirect\Generic\Category\LanguageId;
use DealerDirect\Generic\Category\Locale as LocaleCategory;
use DealerDirect\Generic\Category\Site;
use DealerDirect\Generic\Exception\NotImplementedException;

class LocaleRepository implements RepositoryInterface
{
    /**
     * Returns all possible locales for DealerDirect
     */
    public function getAll()
    {
        throw new NotImplementedException();
    }

    /**
     * Return *supported* locales as defined in DealerDirect\Generic\Category\Locale
     *
     * @return array
     */
    public function getSupportedLocales()
    {
        return array_values(LocaleCategory::getAll());
    }

    /**
     * @param $countryId
     * @param $defaultCountryCode
     *
     * @return mixed
     */
    public function getCountryCodeForCountryId($countryId, $defaultCountryCode = CountryTerritoryCode::NETHERLANDS)
    {
        /** @CHECKME: Should the default be NL or "Unknown Country code" ? */
        $countryCode = $defaultCountryCode;

        $key = Country::getKey($countryId);

        if ($key !== false) {
            $countryCode = constant('\DealerDirect\Generic\Category\CountryTerritoryCode::' . $key);
        }

        return $countryCode;
    }

    /**
     * @param int $languageId DealerDirect Language ID
     * @param string $defaultLanguage Default Language to return if there is no match for given Language ID
     *
     * @return string The language code
     */
    public function getLanguageForId($languageId, $defaultLanguage = Language::NL)
    {
        switch ((int) $languageId) {
            case LanguageId::GERMAN:
                return Language::DE;
            case LanguageId::FRENCH:
                return Language::FR;
            case LanguageId::ENGLISH:
                return Language::EN;
            case LanguageId::DUTCH:
            case LanguageId::FLEMISH:
                return Language::NL;
            default:
                return $defaultLanguage;
        }
    }

    /**
     * @param int $languageId
     * @param int $countryId
     * @param Locale $defaultLocale
     *
     * @return Locale
     */
    public function getLocaleForLanguageIdAndCountryId($languageId, $countryId, Locale $defaultLocale = null)
    {
        if ($defaultLocale === null) {
            $defaultLocale = new Locale('nl', 'NL');
        }
        $countryCode = $this->getCountryCodeForCountryId($countryId);
        $language = $this->getLanguageForId($languageId);
        try {
            $locale = new Locale($language, $countryCode);
        } catch (\InvalidArgumentException $e) {
            $locale = $defaultLocale;
        }

        return $locale;
    }

    /**
     * Return default Locale for given site reference ID.
     *
     * @param int $siteReferenceId Site reference id
     * @param string $site Only used to specify French language, in case of Belgian site
     *
     * @return Locale The locale based on given site reference. Returns nl_NL as default locale.
     */
    public function getForSiteReference($siteReferenceId, $site = '')
    {
        switch ($siteReferenceId) {
            // German
            case Site::REF_CAR_DE:
            case Site::REF_MOTO_DE:
            case Site::REF_SCOOTER_DE:
                return new Locale('de', 'DE');
            // Belgian
            case Site::REF_CAR_BE_NL:
                return new Locale('nl', 'BE');
            case Site::REF_CAR_BE_FR:
                return new Locale('fr', 'BE');
            case Site::REF_MOTO_BE:
            case Site::REF_SCOOTER_BE:
                if (strpos($site, Site::SITE_LANGUAGE_CHECK_FRENCH) !== false) {
                    return new Locale('fr', 'BE');
                }
                return new Locale('nl', 'BE');
            case Site::REF_CAR_FR:
                return new Locale('fr', 'FR');
            // Dutch
            default:
                return new Locale('nl', 'NL');
        }
    }

    /**
     * Return default Locale (as string) for given site reference ID.
     *
     * @param int $siteReferenceId Site reference id
     * @param string $site Only used to specify French language, in case of Belgian site
     *
     * @return string The locale string based on given site reference. Returns nl_NL as default locale.
     */
    public function getForSiteReferenceAsString($siteReferenceId, $site = '')
    {
        switch ($siteReferenceId) {
            // German
            case Site::REF_CAR_DE:
            case Site::REF_MOTO_DE:
            case Site::REF_SCOOTER_DE:
                return LocaleCategory::DE_DE;
            // Belgian
            case Site::REF_CAR_BE_NL:
                return LocaleCategory::NL_BE;
            case Site::REF_CAR_BE_FR:
                return LocaleCategory::FR_BE;
            case Site::REF_MOTO_BE:
            case Site::REF_SCOOTER_BE:
                if (strpos($site, Site::SITE_LANGUAGE_CHECK_FRENCH) !== false) {
                    return LocaleCategory::FR_BE;
                }
                return LocaleCategory::NL_BE;
            case Site::REF_CAR_FR:
                return LocaleCategory::FR_FR;
            // Dutch
            default:
                return LocaleCategory::NL_NL;
        }
    }
}
