<?php
namespace DealerDirect\Generic\Repository;

use DealerDirect\Generic\Category\TranslationTypes;

/**
 * Class Translator
 * @package DealerDirect\Generic\Repository
 *
 * Translator for Vehicle properties
 *
 * Usage example:
 *      //Set Locale and GenericProperties.
 *      $locale = new Locale(Language::NL, Territory::NL);
 *      $genericProperties = new GenericProperties($locale, VehicleType::CAR);
 *
 *      $translator = new Translator($genericProperties);
 *      $translatedValues = $translator->get(TranslationTypes::OPTIONS, [
 *          VehicleTypeCarOptions::AIRCO,
 *          VehicleTypeCarOptions::ABS,
 *          VehicleTypeCarOptions::AIRBAG_PASSENGER,
 *          etc...
 *      ]);
 */
class Translator extends CommonTranslator
{
    /** @var string */
    private $vehicleType;

    /**
     * Translator constructor.
     *
     * @param GenericProperties $genericProperties
     */
    public function __construct(GenericProperties $genericProperties)
    {
        parent::__construct($genericProperties->getLocaleInstance());
        $this->vehicleType = $genericProperties->getVehicleType();
    }

    /**
     * @param $translationsType
     * @return array
     */
    public function getAll($translationsType)
    {
        TranslationTypes::assertExists($translationsType);
        return $this->getTranslations($translationsType, $this->vehicleType);
    }

    /**
     * @param $translationsType
     * @param array|string $ids
     * @param bool $returnWithIds
     *
     * @return array
     *
     * @throws \Exception
     * @throws \InvalidArgumentException
     */
    public function get($translationsType, $ids = [], $returnWithIds = false)
    {
        TranslationTypes::assertExists($translationsType);
        $translations = $this->getTranslations($translationsType, $this->vehicleType);

        // @CHECKME: If `$ids` is a string && $type !== TranslationTypes::OPTIONS this breaks.
        if (is_string($ids) && $translationsType === TranslationTypes::OPTIONS) {
            $ids = $this->getArrayFromOptionsString($ids);
        }

        $results = [];

        foreach ($ids as $id) {
            $value = $this->defaultValue;

            if (array_key_exists($id, $translations)) {
                $value = $translations[$id];
            }

            $results[$id] = $value;
        }

        if ($returnWithIds === false) {
            $results = array_values($results);
        }

        return $results;
    }
    
    /**
     * @param string $ids
     *
     * @return array
     */
    private function getArrayFromOptionsString($ids)
    {
        $ids = trim($ids, ';');
        $ids = explode(';', $ids);
        
        return $ids;
    }
}
