<?php

namespace DealerDirect\Generic\Repository;

use DealerDirect\Generic\Category\VehicleType;
use Doctrine\DBAL\Connection;

class MakesAndModels implements RepositoryInterface
{
    public const CAR_STRUCTURES = '1,2,5';
    public const MOTOR_STRUCTURES = '3,4';
    public const RECREATION_STRUCTURES = '6';
    public const SCOOTER_STRUCTURES = '7';
    public const BICYCLE_STRUCTURES = '8';
    public const ALL_STRUCTURES = '1,2,3,4,5,6,7,8';

    public const STATUS_ACTIVE = 1;

    /** @var Connection */
    protected $dbal;
    /** @var string */
    protected $vehicleType;

    /**
     * MakesAndModels constructor.
     *
     * @param Connection $dbal
     * @param string $vehicleType
     */
    public function __construct(Connection $dbal, $vehicleType)
    {
        VehicleType::assertExists($vehicleType);
        $this->vehicleType = $vehicleType;
        $this->dbal = $dbal;
    }

    /**
     * @return array|null
     */
    public function getAll()
    {
        $results = $this->dbal->fetchAllAssociative(
            $this->getAllMakesModelsQuery(),
            ['status' => self::STATUS_ACTIVE]
        );

        return !empty($results) ? $this->formatResults($results) : null;
    }

    /**
     * @param string $makeName
     *
     * @return int
     */
    public function getMakeIdByName($makeName)
    {
        return (int) $this->dbal->fetchOne(
            <<<SQL
            SELECT merkID
            FROM dealer01_webservices.SB_MERKEN
            WHERE merkNaam = :makeName
            SQL,
            ['makeName' => $this->normalizeMakeName($makeName)]
        );
    }

    /**
     * @param int $makeId
     * @param string $modelName
     *
     * @return int
     */
    public function getModelIdByName($makeId, $modelName)
    {
        return (int) $this->dbal->fetchOne(
            <<<SQL
            SELECT modelID
            FROM dealer01_webservices.SB_MODELLEN
            WHERE modelNaam = :modelName
            AND merkID = :makeId
            SQL,
            ['modelName' => $this->normalizeModelName($makeId, $modelName)]
        );
    }

    /**
     * @param bool $returnArray
     *
     * @return array|string
     */
    protected function getVehicleTypeStructure($returnArray = false)
    {
        $vehicleType = $this->vehicleType;
        switch ($vehicleType) {
            case VehicleType::CAR:
                $result = self::CAR_STRUCTURES;
                break;
            case VehicleType::SCOOTER:
                $result = self::SCOOTER_STRUCTURES;
                break;
            case VehicleType::MOTOR:
                $result = self::MOTOR_STRUCTURES;
                break;
            case VehicleType::BICYCLE:
                $result = self::BICYCLE_STRUCTURES;
                break;
            case VehicleType::RECREATION:
                $result = self::RECREATION_STRUCTURES;
                break;
            case VehicleType::GENERIC:
                $result = self::ALL_STRUCTURES;
                break;
            default:
                $errorMsg = sprintf(
                    'Invalid vehicleType, given vehicleType: "%s", expected one of these vehicleTypes "%s".',
                    $vehicleType,
                    implode(', ', VehicleType::getAll())
                );
                throw new \InvalidArgumentException($errorMsg);
                break;
        }
        if ($returnArray === true) {
            $result = explode(',', $result);
        }

        return $result;
    }

    /**
     * @param array $unsortedResult
     *
     * @return array
     */
    private function formatResults($unsortedResult)
    {
        $models = [];

        foreach ($unsortedResult as $row) {
            if (!array_key_exists($row['makeId'], $models)) {
                $models[$row['makeId']] = [];
                $models[$row['makeId']]['makeId'] = (int) $row['makeId'];
                $models[$row['makeId']]['makeName'] = utf8_encode(trim($row['makeName']));
                $models[$row['makeId']]['models'] = [];
            }
            $model = [
                'modelId' => (int) $row['modelId'],
                'modelName' => utf8_encode(trim($row['modelName'])),
            ];
            array_push($models[$row['makeId']]['models'], $model);
        }

        return $models;
    }

    /**
     * @return string
     */
    private function getAllMakesModelsQuery()
    {
        return <<<SQL
            SELECT
                min(mk.merkID) as makeId,
                min(md.modelID) as modelId,
                IF (rme.naar IS NULL OR rme.van = 'ALPINA', UPPER(mk.merkNaam), UPPER(rme.naar)) makeName,
                IF (rmo.naar IS NULL, UPPER(md.modelNaam), UPPER(rmo.naar)) modelName
            FROM dealer01_webservices.SB_MERKEN mk
            INNER JOIN dealer01_webservices.SB_MODELLEN md
                ON md.merkId = mk.merkId
            LEFT JOIN dealer01_webservices.SB_REGELS_MODELLEN rmo
                ON md.merkID = rmo.merkID
                AND md.modelNaam = rmo.van
            LEFT JOIN (
                select van,naar from dealer01_webservices.SB_REGELS_MERKEN
                group by van,naar
            ) as rme
            ON mk.merkNaam = rme.van
            where md.modelID in (
                select modelID
                from dealer01_webservices.SB_STRUCTUUR_MODELLEN
                where
                    structuurID IN ({$this->getVehicleTypeStructure()})
                    AND active = :status
            )
            GROUP BY makeName, modelName
            ORDER BY makeName ASC, modelName ASC
        SQL;
    }

    /**
     * @param string $makeName
     *
     * @return string
     */
    public function normalizeMakeName($makeName)
    {
        return $this->dbal->fetchOne(
            <<<SQL
            SELECT naar
            FROM dealer01_webservices.SB_REGELS_MERKEN
            WHERE van = :makeName
            SQL,
            compact('makeName'),
        ) ?: $makeName;
    }

    /**
     * @param int $makeId
     * @param string $modelName
     *
     * @return string
     */
    public function normalizeModelName($makeId, $modelName)
    {
        return $this->dbal->fetchOne(
            <<<SQL
            SELECT naar
            FROM dealer01_webservices.SB_REGELS_MODELLEN
            WHERE van = :modelName
            AND merkID = :makeId
            SQL,
            compact('makeId', 'modelName')
        ) ?: $modelName;
    }
}
