<?php

namespace DealerDirect\Generic\Util;

use libphonenumber\PhoneNumberUtil;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberType;
use libphonenumber;

class PhoneNumberFormatting
{
    /**
     * Country Code
     *
     * @var
     */
    private $countryCode;

    /**
     * PhoneNumberFormatting constructor.
     *
     * @param int $countryId Country Id.
     */
    public function __construct($countryId)
    {
        $this->countryCode = $this->getPhoneCountryCode($countryId);
    }

    /**
     * Formats the phone number to the E.164 numbering plan.
     *
     * @param $phoneNumber
     *
     * @return string
     */
    public function formatPhoneNumberToE164($phoneNumber)
    {
        return $this->formatPhoneNumber(PhoneNumberFormat::E164, $phoneNumber);
    }

    /**
     * Formats the phone number to the National numbering plan.
     *
     * @param $phoneNumber
     *
     * @return string
     */
    public function formatPhoneNumberToNational($phoneNumber)
    {
        return $this->formatPhoneNumber(
            PhoneNumberFormat::NATIONAL,
            $phoneNumber
        );
    }

    /**
     * Check if the given phone number is a mobile number
     *
     * @param int $phoneNumber Phone Number
     *
     * @return bool
     */
    public function isPhoneNumberMobile($phoneNumber)
    {
        $phoneUtil = PhoneNumberUtil::getInstance();
        if (empty($phoneNumber)) {
            return false;
        }
        try {
            $parsedPhone = $phoneUtil->parse($phoneNumber, $this->countryCode);
            $isNumberType = $phoneUtil->getNumberType($parsedPhone);
            if ($isNumberType === PhoneNumberType::MOBILE) {
                return true;
            }
        } catch (libphonenumber\NumberParseException $e) {
            return false;
        }
        return false;
    }

    /**
     * Function to parse a given phone number to a given format.
     *
     * @param string    $numberFormat
     * @param int       $phoneNumber
     *
     * @return string
     */
    private function formatPhoneNumber($numberFormat, $phoneNumber)
    {
        $phoneUtil = PhoneNumberUtil::getInstance();
        if (empty($phoneNumber)) {
            return null;
        }
        try {
            $parsedPhone = $phoneUtil->parse($phoneNumber, $this->countryCode);
            $isValid = $phoneUtil->isValidNumber($parsedPhone);
            if ($isValid) {
                return $phoneUtil->format($parsedPhone, $numberFormat);
            }
        } catch (libphonenumber\NumberParseException $e) {
            return null;
        }
        return null;
    }

    /**
     * Returns the country code based on the country Id
     *
     * @param int $countryId Country Id
     *
     * @return string
     */
    private function getPhoneCountryCode($countryId)
    {
        switch ((int)$countryId) {
            case '1':
                $countryCode = 'NL';
                break;
            case '2':
                $countryCode = 'BE';
                break;
            case '3':
                $countryCode = 'DE';
                break;
            default:
                $countryCode = 'NL';
        }
        return $countryCode;
    }
}
