<?php
namespace DealerDirect\Generic\Repository;

use DealerDirect\Generic\Category\TranslationTypes;

/**
 * Class Translator
 * @package DealerDirect\Generic\Repository
 *
 * Usage example:
 *      //Set Locale and GenericProperties.
 *      $locale = new Locale(Language::NL, Territory::NL);
 *      $genericProperties = new GenericProperties($locale, VehicleType::CAR);
 *
 *      $translator = new Translator($genericProperties);
 *      $translatedValues = $translator->get(TranslationTypes::OPTIONS, [
 *          VehicleTypeCarOptions::AIRCO,
 *          VehicleTypeCarOptions::ABS,
 *          VehicleTypeCarOptions::AIRBAG_PASSENGER,
 *          etc...
 *      ]);
 */
class Translator {
    
    /** @var string */
    private $locale;
    /** @var string */
    private $vehicleType;
    /** @var string */
    private $defaultValue;
    
    /**
     * Translator constructor.
     *
     * @param GenericProperties $genericProperties
     */
    public function __construct(GenericProperties $genericProperties)
    {
        $this->locale = strtolower($genericProperties->getLocale());
        $this->vehicleType = $genericProperties->getVehicleType();
    }
    
    public function getAll($type)
    {
        TranslationTypes::assertExists($type);
        $translations = $this->getLangFile($type);
        
        return $translations;
    }
    
    /**
     * @param $type
     * @param array|string $ids
     * @param bool $returnWithIds
     *
     * @return array
     */
    public function get($type, $ids = [], $returnWithIds = false)
    {
        TranslationTypes::assertExists($type);
        $translations = $this->getLangFile($type);
        if (is_string($ids) && $type === TranslationTypes::OPTIONS) {
            $ids = $this->getArrayFromOptionsString($ids);
        }
        $results = [];
        foreach ($ids as $id)
        {
            $value = $this->defaultValue;
            if (array_key_exists($id, $translations)) {
                $value = $translations[$id];
            }
            if ($returnWithIds) {
                $results[$id] = $value;
            } else {
                $results[] = $value;
            }
        }
        
        return $results;
    }
    
    /**
     * @param $type
     *
     * @return array
     * @throws \Exception
     */
    private function getLangFile($type)
    {
        $langFile = __DIR__ . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'Lang' . DIRECTORY_SEPARATOR
            . $this->locale . DIRECTORY_SEPARATOR . $type;
        if (!file_exists($langFile)) {
            $errorMsg = sprintf('Error, could not load translations file. Given path: "%s"', $langFile);
            throw new \Exception($errorMsg);
        }
        $file = include $langFile;
        $vehicleType  = $this->vehicleType;
        if (!is_array($file) || !array_key_exists($vehicleType, $file)) {
            $errorMsg = sprintf('Error, could not find vehicleType in translations');
            throw new \Exception($errorMsg);
        }
        $this->defaultValue = $file['default'];
    
        return $file[$vehicleType];
    }
    
    /**
     * @param string $ids
     *
     * @return array
     */
    private function getArrayFromOptionsString($ids)
    {
        $ids = trim($ids, ';');
        $ids = explode(';', $ids);
        
        return $ids;
    }
}
