<?php

namespace DealerDirect\Generic\Util;

use Exception;
use GuzzleHttp\Client;

class ExactApiWrapper
{
    /**
     * @var Client
     */
    private $client;

    /**
     * @var string
     */
    private $baseUrl;

    /**
     * @var string
     */
    private $accessToken;

    /**
     * @param string $baseUrl
     * @param int $clientId
     * @param string $clientSecret
     */
    public function __construct(string $baseUrl, int $clientId, string $clientSecret)
    {
        $this->client = new Client();
        $this->baseUrl = $baseUrl;
        $this->accessToken = $this->getAccessToken($clientId, $clientSecret);
    }

    /**
     * @param integer $companyId
     * @param integer $lotId
     * @param integer $invoiceUnitPrice
     * @param string $invoiceDescription
     * @param string $invoiceVehicleType
     * @param string $invoiceCountry
     * @param string $invoiceReference
     * @param integer $invoiceQuantity
     * @return mixed
     */
    public function createInvoice(
        int $companyId,
        int $lotId,
        int $invoiceUnitPrice,
        string $invoiceDescription,
        string $invoiceVehicleType,
        string $invoiceCountry,
        string $invoiceReference = 'exact_invoice',
        int $invoiceQuantity = 1
    ) {
        $response = $this->client->post("{$this->baseUrl}/api/invoice/create", [
            'headers' => ['Authorization' => "Bearer {$this->accessToken}"],
            'form_params' => [
                'companycode' => $companyId,
                'invoiceReference' => $invoiceReference,
                'invoiceQuantity' => $invoiceQuantity,
                'invoiceUnitPrice' => $invoiceUnitPrice,
                'invoiceDescription' => $invoiceDescription,
                'invoiceVehicleType' => $invoiceVehicleType,
                'invoiceCountry' => $invoiceCountry,
                'lotId' => $lotId,
            ],
        ]);

        return json_decode($response->getBody()->getContents(), true);
    }

    /**
     * @param int $clientId
     * @param string $clientSecret
     * @return string
     */
    private function getAccessToken(int $clientId, string $clientSecret)
    {
        $response = $this->client->post("{$this->baseUrl}/oauth/token", [
            'form_params' => [
                'grant_type' => 'client_credentials',
                'client_id' => $clientId,
                'client_secret' => $clientSecret,
            ]
        ]);

        $contents = json_decode($response->getBody()->getContents());
        if (empty($contents->access_token)) {
            throw new Exception('Error, could not get accessToken from exact-api');
        }

        return $contents->access_token;
    }
}
