<?php

namespace Dealerdirect\Generic\Util\HealthCheck;

use Closure;
use Dealerdirect\Generic\Category\ResponseCode;
use Dealerdirect\Generic\Category\ResponseType;
use Exception;

class HealthCheck
{
    ////////////////////////////// CLASS PROPERTIES \\\\\\\\\\\\\\\\\\\\\\\\\\\\

    const MESSAGE_FAILURE = 'an unknown error occurred';
    const MESSAGE_SUCCESS = 'all systems up and running';

    /** @var string */
    private $contentType;
    /** @var string */
    private $response = self::MESSAGE_FAILURE;
    /** @var int */
    private $responseCode;

    //////////////////////////// SETTERS AND GETTERS \\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @return string
     */
    final public function getResponse()
    {
        http_response_code($this->responseCode);
        header('Content-Type: text/'.$this->contentType);
        return $this->response;
    }

    /**
     * Callback to be registered using PHP's native `register_shutdown_function`
     *
     * @return Closure
     *
     * @suppress PhanUnreferencedMethod
     */
    public function getShutdownFunction()
    {
        return function () {
            $this->parseLastError();
        };
    }

    //////////////////////////////// PUBLIC API \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @param string $responseType
     * @param Closure $check
     *
     * @return string
     *
     * @throws \InvalidArgumentException
     */
    final public function run($responseType, Closure $check)
    {
        ResponseType::assertExists($responseType);

        register_shutdown_function($this->getShutdownFunction());

        $this->runCheck($responseType, $check);

        return $this->getResponse();
    }

    /**
     * @param string $type
     * @param Closure $check
     */
    protected function runCheck($type, Closure $check)
    {
        $response = self::MESSAGE_FAILURE;
        $responseCode = ResponseCode::HTTP_INTERNAL_SERVER_ERROR;

        $this->contentType = $type;

        http_response_code($responseCode);

        /* Make sure we can remove any output caused by the health check */
        ob_start();

        try {
            $check();
        } catch (Exception $exception) {
            $response = $this::sanitizeException($exception);
        }

        if (is_array($response) === false) {
            $contents = trim(strip_tags(ob_get_clean()));

            $response = self::MESSAGE_SUCCESS;
            $responseCode = ResponseCode::HTTP_OK;

            if ($contents !== '') {
                $responseCode = ResponseCode::HTTP_INTERNAL_SERVER_ERROR;
                $response = $this::buildErrorFromString($contents);
            }
        }

        $this->responseCode = $responseCode;
        $this->response = $response;
    }

    /**
     * Create array from exception
     *
     * @param Exception $exception
     *
     * @return array
     */
    public static function sanitizeException(Exception $exception)
    {
        $response = [];

        $exceptionArray = (array)$exception;

        $exceptionArray['type'] = get_class($exception);

        $ignore = ['xdebug_message', 'trace'];

        /* @NOTE: Path within package repository */
        $path = dirname(dirname(__DIR__)) . '/';

        if (strpos($path, '/vendor/')) {
            /* @NOTE: Path of project using the package */
            $path = str_replace('vendor/dealerdirect/generic/src/', '', $path);
        }

        array_walk($exceptionArray, function ($value, $key) use (&$response, $ignore, $path) {

            /* Clean up key */
            $keyParts = explode("\0", $key);
            $sanitizedKey = array_pop($keyParts);

            /* Remove directory name from value */

            $sanitizedValue = str_replace($path . '/', '', $value);

            if (in_array($sanitizedKey, $ignore, true) === false) {
                /* Ignore values from the ignore list */
                $response[$sanitizedKey] = $sanitizedValue;
            }
        });

        /* Remove empty values */
        $response = array_filter($response);

        $previous = $exception->getPrevious();

        if (array_key_exists('message', $response) === false
            && $previous instanceof Exception
            &&  (string) $previous->getMessage() !== ''
        ) {
            $response['message'] = $previous->getMessage();
        }

        if (array_key_exists('message', $response) === false || (string) $response['message'] === '') {
            $response['message'] = self::MESSAGE_FAILURE;
        }

        return $response;
    }

    ////////////////////////////// UTILITY METHODS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    private function parseLastError()
    {
        $lastError = error_get_last();

        $allowed = [
            E_DEPRECATED,
        ];

        if (is_array($lastError) && in_array($lastError['type'], $allowed, true) === false) {
            ob_clean();
            $this->responseCode = ResponseCode::HTTP_INTERNAL_SERVER_ERROR;
            $this->response = $this->format($lastError);

            echo $this->getResponse();
        }
    }

    /**
     * @param array $lastError
     *
     * @return string
     */
    private function format(array $lastError)
    {
        $unknownValue = '';

        switch ($this->contentType) {

            case ResponseType::NU_SOAP:
            case ResponseType::SOAP:
                $format = <<<'XML'
<SOAP-ENV:Envelope 
    SOAP-ENV:encodingStyle="http://schemas.xmlsoap.org/soap/encoding/" 
    xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/" 
    xmlns:xsd="http://www.w3.org/2001/XMLSchema" 
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" 
    xmlns:SOAP-ENC="http://schemas.xmlsoap.org/soap/encoding/"
>
   <SOAP-ENV:Body>
      <ns1:healthCheckResponse xmlns:ns1="urn:healthCheck">
         <return>
            <message xsi:type="xsd:string">%s</message>
            <file xsi:type="xsd:string">%s</file>
            <line xsi:type="xsd:int">%s</line>
            <type xsi:type="xsd:int">%s</type>
         </return>
      </ns1:healthCheckResponse>
   </SOAP-ENV:Body>
</SOAP-ENV:Envelope>
XML;
                break;

            case ResponseType::REST:
                $format = <<<'JSON'
{
  "code": 500,
  "data": {
  },
  "error": {
    "message": "%s",
    "file": "%s",
    "line": %d,
    "type": %d
  },
  "message": "Internal Server Error",
  "status": "error"
}
JSON;
                $unknownValue = 'undefined';
                break;

            case ResponseType::TEXT:
            default:
                $format = <<<'TXT'
MESSAGE: AN ERROR OCCURRED
ERROR: %s
FILE: %s
LINE: %s
TYPE: %s
TXT;
                $unknownValue = '-- unknown --';
                break;
        }

        $args = [
            isset($lastError['message']) ? $lastError['message'] : self::MESSAGE_FAILURE,
            isset($lastError['file']) ? $lastError['file'] : $unknownValue,
            isset($lastError['line']) ? $lastError['line'] : $unknownValue,
            isset($lastError['type']) ? $lastError['type'] : $unknownValue,
        ];

        return vsprintf($format, $args);
    }

    /**
     * @param $contents
     *
     * @return mixed
     */
    final public static function buildErrorFromString($contents)
    {
        $response = self::MESSAGE_FAILURE;

        /* @NOTE Parsing for errors caused by `mysql_connect()`, other errors might creep pass. */
        $pattern = '%(?P<type>[^:]+):\s+(?P<message>.+)\s+in\s+(?P<file>.+?)\s+on\s+line\s+(?P<line>[0-9]+)%i';

        preg_match($pattern, $contents, $lastError);

        array_walk($lastError, function ($value, $key) use (&$lastError) {
            if (is_numeric($key)) {
                unset($lastError[$key], $value);
            }
        });

        if (count($lastError) > 0) {
            $response = $lastError;
        }

        return $response;
    }
}
/*EOF*/
