<?php

namespace DealerDirect\Generic\Repository;

use DealerDirect\Generic\Category\Locale as LocaleCategory;
use DealerDirect\Generic\Category\Territory;
use DealerDirect\Generic\Category\Language;

/**
 * Immutable object holding the value of an RFC-5646 compliant "Tag for Identifying Languages"
 *
 * Also known as a "Locale" the form of such a tag is:
 *
 *     language[_territory][.codeset][@modifier]
 *
 * Where:
 * - language is a two-letter, lowercase and ISO_639-1 compliant
 * - territory is two-letter, uppercase and ISO 3166-1 alpha-2 compliant
 *
 * Despite the RFC dictating values as *case insensitive*, casing conventions are
 * upheld for the sake of being recognisable as values used elsewhere.
 */
class Locale implements RepositoryInterface
{
    ////////////////////////////// CLASS PROPERTIES \\\\\\\\\\\\\\\\\\\\\\\\\\\\
    /** @var string */
    private $language;
    /** @var string */
    private $territory;

    //////////////////////////// SETTERS AND GETTERS \\\\\\\\\\\\\\\\\\\\\\\\\\\
    /** @return string */
    final public function getLanguage()
    {
        return $this->language;
    }

    /** @return string */
    final public function getTerritory()
    {
        return $this->territory;
    }

    //////////////////////////////// PUBLIC API \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
    /**
     * Locale constructor.
     *
     * @param string $language
     * @param string $territory
     *
     * @throws \InvalidArgumentException
     *
     * @CHECKME: Add Category classes for codeset/modifier? They are not used (and it is not certain they ever will be...) - 2016/09/12/BMP
     */
    final public function __construct($language, $territory = ''/*, $codeset = '', $modifier = ''*/)
    {
        $language = mb_strtolower($language);
        $territory = mb_strtoupper($territory);

        Language::assertExists($language);

        if ($territory !== '') {
            Territory::assertExists($territory);
        }

        $this->language = $language;
        $this->territory = $territory;

        LocaleCategory::assertExists((string) $this);
    }

    /** @return string */
    final public function __toString()
    {
        $locale = $this->getLanguage();

        if ($this->getTerritory() !== '') {
            $locale .= '_' . $this->getTerritory();
        }

        return $locale;
    }
}
