<?php

namespace DealerDirect\Generic\Config;

/**
 * Abstract Config class
 *
 * This class is meant to be extend by projects to hold project specific configuration key/value pairs
 */
abstract class AbstractConfig implements ConfigInterface
{
    public const ERROR_KEY_NOT_SUPPORTED = 'Key(s) "%s" are not-supported';

    /** @var string[] */
    private $config = [];

    /**
     * Array of keys that the Config object can contain
     *
     * @return string[]
     */
    abstract public function getKeys();

    /**
     * Creates an Config object from an array of key/value pairs.
     *
     * Any values that have a key which the Config object is not familiar with
     * are not set in the config but will simply be ignored.
     *
     * All keys must be of type "string". Numeric keys are NOT supported and will
     * trigger an exception.
     *
     * @param string[] $config
     *
     * @throws \DealerDirect\Generic\Config\ConfigException
     */
    public function __construct(array $config)
    {
        $this->setValues($config);
    }

    /**
     * Validate if a given key exists and throws an exception if it does not exist
     *
     * To validate if a given key exists without throwing an exception, use the `exist` method
     *
     * @param string $key
     *
     * @return void
     *
     * @throws ConfigException Given key does not exist
     */
    public function assertExist($key)
    {
        if ($this->exist($key) === false) {
            $message = vsprintf(self::ERROR_UNKNOWN_KEY, [$key, implode('", "', $this->getKeys())]);
            throw new ConfigException($message);
        }
    }

    /**
     * Validate whether a given key exists or not
     *
     * @param string $key
     *
     * @return bool
     */
    public function exist($key)
    {
        return in_array($key, $this->getKeys(), true);
    }

    /**
     * Gets the value for a given key or throws an exception if it does not exist
     *
     * @param string $key
     *
     * @return mixed Value for given key
     *
     * @throws ConfigException Given key is not set
     */
    public function get($key)
    {
        if (in_array($key, $this->getKeys(), true) === false) {
            $message = vsprintf(self::ERROR_KEY_NOT_SUPPORTED, [$key]);
            throw new ConfigException($message);
        }

        if (array_key_exists($key, $this->config) === false) {
            $message = vsprintf(self::ERROR_GET_BEFORE_SET, [$key]);
            throw new ConfigException($message);
        }

        return $this->config[$key];
    }

    /**
     * @param array $config
     *
     * @throws \DealerDirect\Generic\Config\ConfigException
     */
    private function setValues(array $config)
    {
        array_walk($config, function ($value, $key) {
            if (is_string($key) === false) {
                $message = vsprintf(self::ERROR_KEY_MUST_BE_STRING, [$value, gettype($key)]);
                throw new ConfigException($message);
            }

            $exist = $this->exist($key);
            if ($exist === true) {
                $this->config[$key] = $value;
            }/* Unknown values are simply ignored */
        });
    }
}

/*EOF*/
