<?php
namespace DealerDirect\Generic\Repository;

require_once __DIR__ . '/../../vendor/autoload.php';

use DealerDirect\Generic\Category\TranslationTypes;

/**
 * Class Translator
 * @package DealerDirect\Generic\Repository
 *
 * Usage example:
 *      //Set Locale and GenericProperties.
 *      $locale = new Locale(Language::NL, Territory::NL);
 *      $genericProperties = new GenericProperties($locale, VehicleType::CAR);
 *
 *      $translator = new Translator($genericProperties);
 *      $translatedValues = $translator->get(TranslationTypes::OPTIONS, [
 *          VehicleTypeCarOptions::AIRCO,
 *          VehicleTypeCarOptions::ABS,
 *          VehicleTypeCarOptions::AIRBAG_PASSENGER,
 *          etc...
 *      ]);
 */
class Translator {
    
    const LANG_PATH = __DIR__ . '/../Lang/';
    
    /** @var string */
    private $locale;
    /** @var string */
    private $vehicleType;
    /** @var string */
    private $defaultValue;
    
    /**
     * Translator constructor.
     *
     * @param GenericProperties $genericProperties
     */
    public function __construct(GenericProperties $genericProperties)
    {
        $this->locale = strtolower($genericProperties->getLocale());
        $this->vehicleType = $genericProperties->getVehicleType();
    }
    
    public function getAll($type)
    {
        TranslationTypes::assertExists($type);
        $translations = $this->getLangFile($type);
        
        return $translations;
    }
    
    /**
     * @param $type
     * @param array|string $ids
     *
     * @return array
     */
    public function get($type, $ids = [])
    {
        TranslationTypes::assertExists($type);
        $translations = $this->getLangFile($type);
        if (is_string($ids) && $type === TranslationTypes::OPTIONS) {
            $ids = $this->getArrayFromOptionsString($ids);
        }
        
        $results = [];
        foreach ($ids as $id)
        {
            $value = $this->defaultValue;
            if (array_key_exists($id, $translations)) {
                $value = $translations[$id];
            }
            $results[] = $value;
        }
        
        return $results;
    }
    
    /**
     * @param $type
     *
     * @return array
     * @throws \Exception
     */
    private function getLangFile($type)
    {
        $langFile = self::LANG_PATH . $this->locale . '/' . $type;
        if (!file_exists($langFile)) {
            $errorMsg = sprintf('Error, could not load translations file. Given path: "%s"', $langFile);
            throw new \Exception($errorMsg);
        }
        $file = include $langFile;
        $vehicleType  = $this->vehicleType;
        if (!is_array($file) || !array_key_exists($vehicleType, $file)) {
            $errorMsg = sprintf('Error, could not find vehicleType in translations');
            throw new \Exception($errorMsg);
        }
        $this->defaultValue = $file['default'];
    
        return $file[$vehicleType];
    }
    
    /**
     * @param string $ids
     *
     * @return array
     */
    private function getArrayFromOptionsString($ids)
    {
        $ids = trim($ids, ';');
        $ids = explode(';', $ids);
        
        return $ids;
    }
}
