<?php

namespace Dealerdirect\Generic\VehicleInformation;

use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Category\VehicleTypeCarOptions;
use DealerDirect\Generic\Category\VehicleTypeMotorOptions;
use DealerDirect\Generic\Category\VehicleTypeRecreationalOptions;
use Exception;

class VehicleOptions
{
    private const MOST_FREQUENT = [
        VehicleType::CAR => [
            VehicleTypeCarOptions::AIRCO,
            VehicleTypeCarOptions::AUTOMATIC_AIRCO,
            VehicleTypeCarOptions::ELECTRIC_WINDOWS,
            VehicleTypeCarOptions::POWER_STEERING,
            VehicleTypeCarOptions::METALLIC_RIMS,
            VehicleTypeCarOptions::CENTRAL_LOCKING,
            VehicleTypeCarOptions::PANORAMIC_ROOF,
            VehicleTypeCarOptions::NAVIGATION_SYSTEM,
            VehicleTypeCarOptions::XENON_LIGHTS,
            VehicleTypeCarOptions::FOUR_WHEEL_DRIVE,
            VehicleTypeCarOptions::CRUISE_CONTROL,
            VehicleTypeCarOptions::ROOF_RAILS,
            VehicleTypeCarOptions::OPEN_ROOF,
            VehicleTypeCarOptions::ALARM,
        ],
        VehicleType::MOTOR => [
            VehicleTypeMotorOptions::ABS,
            VehicleTypeMotorOptions::TRACTION_CONTROL,
            VehicleTypeMotorOptions::CRUISE_CONTROL,
            VehicleTypeMotorOptions::ACTIVE_SUSPENSION,
            VehicleTypeMotorOptions::LUGGAGE_SYSTEM,
            VehicleTypeMotorOptions::ALARM,
            VehicleTypeMotorOptions::HANDLE_HEATING,
            VehicleTypeMotorOptions::SEAT_HEATING,
            VehicleTypeMotorOptions::FALL_BRACES,
            VehicleTypeMotorOptions::WINDSHIELD,
            VehicleTypeMotorOptions::EXHAUST_SYSTEM_ORIGINAL,
            VehicleTypeMotorOptions::EXHAUST_SYSTEM_AFTERMARKET,
        ],
        VehicleType::SCOOTER => [
            VehicleTypeMotorOptions::ABS,
            VehicleTypeMotorOptions::LUGGAGE_SYSTEM,
            VehicleTypeMotorOptions::ALARM,
            VehicleTypeMotorOptions::FALL_BRACES,
            VehicleTypeMotorOptions::WINDSHIELD,
            VehicleTypeMotorOptions::EXHAUST_SYSTEM_ORIGINAL,
            VehicleTypeMotorOptions::EXHAUST_SYSTEM_AFTERMARKET,
        ],
        VehicleType::RECREATION => [
            VehicleTypeRecreationalOptions::BOILER,
            VehicleTypeRecreationalOptions::MOVER,
            VehicleTypeRecreationalOptions::HEATER,
            VehicleTypeRecreationalOptions::TV_ANTENNA,
            VehicleTypeRecreationalOptions::WATER_TANK,
            VehicleTypeRecreationalOptions::ROOF_HATCH,
            VehicleTypeRecreationalOptions::SHOWER,
            VehicleTypeRecreationalOptions::MICROWAVE,
            VehicleTypeRecreationalOptions::ROUNDSEAT,
            VehicleTypeRecreationalOptions::TOILET,
            VehicleTypeRecreationalOptions::FIXED_BED,
            VehicleTypeRecreationalOptions::AWNING,
            VehicleTypeRecreationalOptions::BICYCLE_CARRIER,
            VehicleTypeRecreationalOptions::MIDDLE_KITCHEN,
            VehicleTypeRecreationalOptions::SERVICE_HATCH,
            VehicleTypeRecreationalOptions::TRAIN_SEAT,
        ],
    ];

    private const BRAND_SPECIFIC = [
        'LAND-ROVER' => [
            VehicleTypeCarOptions::LAND_ROVER_DYNAMIC,
        ],
        'MINI' => [
            VehicleTypeCarOptions::MINI_JCW,
        ],
        'MERCEDES-BENZ' => [
            VehicleTypeCarOptions::MERCEDES_AMG,
            VehicleTypeCarOptions::MERCEDES_WIDESCREEN,
            VehicleTypeCarOptions::AIR_SUSPENSION
        ],
        'CITROEN' => [
            VehicleTypeCarOptions::CITROEN_SHINE,
            VehicleTypeCarOptions::CITROEN_CROSSBACK,
        ],
        'FORD' => [
            VehicleTypeCarOptions::FORD_ACTIVE,
            VehicleTypeCarOptions::FORD_ST_LINE,
            VehicleTypeCarOptions::FORD_S_EDITION,
            VehicleTypeCarOptions::FORD_VIGNALE,
            VehicleTypeCarOptions::FORD_TITANIUM,
            VehicleTypeCarOptions::FORD_WINDSHIELD_HEATING,
        ],
        'OPEL' => [
            VehicleTypeCarOptions::OPEL_OPC,
            VehicleTypeCarOptions::OPEL_TWIN_TOP,
            VehicleTypeCarOptions::OPEL_GS_LINE,
            VehicleTypeCarOptions::OPEL_COSMO,
            VehicleTypeCarOptions::OPEL_INNOVATION,
            VehicleTypeCarOptions::OPEL_ROCKS,
            VehicleTypeCarOptions::OPEL_SPORT,
        ],
        'FIAT' => [
            VehicleTypeCarOptions::FIAT_SPORT
        ],
        'AUDI' => [
            VehicleTypeCarOptions::AUDI_S_LINE,
            VehicleTypeCarOptions::VIRTUAL_COCKPIT,
        ],
        'BMW' => [
            VehicleTypeCarOptions::BMW_GRAN_TURISMO,
            VehicleTypeCarOptions::BMW_M_PACKAGE,
            VehicleTypeCarOptions::BMW_SPORT_SEATS,
        ],
        'VOLKSWAGEN' => [
            VehicleTypeCarOptions::VOLKSWAGEN_GTD,
            VehicleTypeCarOptions::VOLKSWAGEN_GTI,
            VehicleTypeCarOptions::VOLKSWAGEN_R_LINE,
            VehicleTypeCarOptions::VOLKSWAGEN_HIGHLINE,
            VehicleTypeCarOptions::VOLKSWAGEN_ALLSPACE,
            VehicleTypeCarOptions::VOLKSWAGEN_ALLTRACK,
            VehicleTypeCarOptions::VOLKSWAGEN_HIGH_UP,
            VehicleTypeCarOptions::VOLKSWAGEN_MOVE_UP,
            VehicleTypeCarOptions::VOLKSWAGEN_TAKE_UP,
            VehicleTypeCarOptions::VIRTUAL_COCKPIT,
            VehicleTypeCarOptions::VOLKSWAGEN_BLACK_STYLE,
            VehicleTypeCarOptions::VOLKSWAGEN_STYLE,
        ],
        'KIA' => [
            VehicleTypeCarOptions::KIA_GT_LINE,
            VehicleTypeCarOptions::KIA_PHEV,
            VehicleTypeCarOptions::KIA_EXECUTIVE_LINE,
        ],
        'TESLA' => [
            VehicleTypeCarOptions::TESLA_AUTOPILOT,
        ],
        'RENAULT' => [
            VehicleTypeCarOptions::RENAULT_RS,
            VehicleTypeCarOptions::RENAULT_HELLY_HANSEN,
            VehicleTypeCarOptions::RENAULT_GT_LINE,
            VehicleTypeCarOptions::RENAULT_BOSE,
            VehicleTypeCarOptions::RENAULT_INTENS,
            VehicleTypeCarOptions::RENAULT_INITIALE,
        ],
        'DACIA' => [
            VehicleTypeCarOptions::DACIA_STEPWAY,
        ],
        'NISSAN' => [
            VehicleTypeCarOptions::NISSAN_TEKNA,
            VehicleTypeCarOptions::NISSAN_NISMO,
            VehicleTypeCarOptions::NISSAN_N_DESIGN,
            VehicleTypeCarOptions::NISSAN_BLACK_EDITION,
            VehicleTypeCarOptions::NISSAN_N_SPORT,
        ],
        'SEAT' => [
            VehicleTypeCarOptions::SEAT_FR,
        ],
        'PEUGEOT' => [
            VehicleTypeCarOptions::PEUGEOT_GT_LINE,
            VehicleTypeCarOptions::PEUGEOT_ALLURE,
            VehicleTypeCarOptions::PEUGEOT_RXH,
            VehicleTypeCarOptions::PEUGEOT_TOP,
        ],
        'SUZUKI' => [
            VehicleTypeCarOptions::SUZUKI_HIGH_EXECUTIVE,
            VehicleTypeCarOptions::SUZUKI_ALL_GRIP,
        ],
        'VOLVO' => [
            VehicleTypeCarOptions::VOLVO_CROSS_COUNTRY,
            VehicleTypeCarOptions::VOLVO_SUMMUM,
            VehicleTypeCarOptions::VOLVO_R_DESIGN,
            VehicleTypeCarOptions::VOLVO_INSCRIPTION,
        ],
        'PORSCHE' => [
            VehicleTypeCarOptions::PORSCHE_SPORT_CHRONO,
        ],
        'LEXUS' => [
            VehicleTypeCarOptions::LEXUS_F_SPORT,
            VehicleTypeCarOptions::LEXUS_PRESIDENT_LINE,
            VehicleTypeCarOptions::LEXUS_LUXURY_LINE,
        ],
        'MAZDA' => [
            VehicleTypeCarOptions::MAZDA_GT_M,
            VehicleTypeCarOptions::MAZDA_SIGNATURE,
        ],
        'ALFA-ROMEO' => [
            VehicleTypeCarOptions::ALFA_ROMEO_TI,
            VehicleTypeCarOptions::ALFA_ROMEO_Q4,
        ],
        'HYUNDAI' => [
            VehicleTypeCarOptions::HYUNDAI_I_CATCHER,
            VehicleTypeCarOptions::HYUNDAI_PHEV,
            VehicleTypeCarOptions::HYUNDAI_N_LINE,
        ],
        'SAAB' => [
            VehicleTypeCarOptions::SAAB_AERO,
        ],
        'JEEP' => [
            VehicleTypeCarOptions::JEEP_SRT
        ],
        'SKODA' => [
            VehicleTypeCarOptions::SKODA_RS,
            VehicleTypeCarOptions::SKODA_MONTE_CARLO,
            VehicleTypeCarOptions::SKODA_BLACK_EDITION,
            VehicleTypeCarOptions::SKODA_SPORTLINE
        ],
        'TOYOTA' => [
            VehicleTypeCarOptions::TOYOTA_BI_TONE,
            VehicleTypeCarOptions::TOYOTA_EXECUTIVE,
            VehicleTypeCarOptions::TOYOTA_X_PLAY,
            VehicleTypeCarOptions::TOYOTA_X_JOY,
            VehicleTypeCarOptions::TOYOTA_GR_SPORT,
        ],
    ];

    private const COMPANY_VEHICLE = [
        VehicleTypeCarOptions::BUMPERS_IN_COLOR,
        VehicleTypeCarOptions::CRUISE_CONTROL,
        VehicleTypeCarOptions::COMPANY_CAR_PANELING,
        VehicleTypeCarOptions::SIDE_BARS,
        VehicleTypeCarOptions::COMPANY_CAR_ROOF_RACK,
        VehicleTypeCarOptions::COMPANY_CAR_SLIDING_DOOR,
        VehicleTypeCarOptions::COMPANY_CAR_REFRIGERATED_TRUCK,
        VehicleTypeCarOptions::COMPANY_CAR_BOX_TRUCK,
        VehicleTypeCarOptions::ELECTRIC_TAILGATE,
        VehicleTypeCarOptions::COMPANY_CAR_DRIVE_ON_TRUCK,
        VehicleTypeCarOptions::COMPANY_CAR_L1,
        VehicleTypeCarOptions::COMPANY_CAR_L2,
        VehicleTypeCarOptions::COMPANY_CAR_L3,
        VehicleTypeCarOptions::COMPANY_CAR_L4,
        VehicleTypeCarOptions::COMPANY_CAR_H1,
        VehicleTypeCarOptions::COMPANY_CAR_H2,
        VehicleTypeCarOptions::COMPANY_CAR_H3,
        VehicleTypeCarOptions::COMPANY_CAR_H4,
    ];

    public function __construct(
        private string $vehicleType,
        private ?string $brand,
        private bool $companyVehicle
    ) {
        $this->brand = strtoupper($brand);
        VehicleType::assertExists($vehicleType);
        if ($vehicleType === VehicleType::CAMPER || $vehicleType === VehicleType::CARAVAN) {
            $this->vehicleType = VehicleType::RECREATION;
        }
    }

    public function getForConsumerSites(): array
    {
        $data = array_merge(
            $this->getMostFrequent(),
            $this->getForBrand(),
            $this->getForCompanyVehicle()
        );
        array_unique($data);

        return $data;
    }

    public function getForBackoffice(): array
    {
        $allOptions = $this->getAll();
        if (in_array($this->vehicleType, [VehicleType::SCOOTER, VehicleType::MOTOR])) {
            // Early exit for scooter/motor as they don't have specific brand options
            return $allOptions;
        }

        $keep = [
            VehicleTypeCarOptions::VIRTUAL_COCKPIT,
            VehicleTypeCarOptions::AIR_SUSPENSION,
            VehicleTypeCarOptions::BUMPERS_IN_COLOR,
            VehicleTypeCarOptions::CRUISE_CONTROL,
            VehicleTypeCarOptions::SIDE_BARS,
            VehicleTypeCarOptions::ELECTRIC_TAILGATE
        ];

        $brandSpecific = self::BRAND_SPECIFIC;
        unset($brandSpecific[$this->brand]);
        foreach ($brandSpecific as $optionsToRemove) {
            foreach ($optionsToRemove as $optionToRemove) {
                if (!in_array($optionToRemove, $keep)) {
                    unset($allOptions[array_search($optionToRemove, $allOptions, true)]);
                }
            }
        }

        if (!$this->companyVehicle) {
            foreach (self::COMPANY_VEHICLE as $optionToRemove) {
                if (!in_array($optionToRemove, $keep)) {
                    unset($allOptions[array_search($optionToRemove, $allOptions, true)]);
                }
            }
        }

        return $allOptions;
    }

    private function getAll(): array
    {
        if ($this->vehicleType === VehicleType::CAR) {
            return VehicleTypeCarOptions::getAll();
        } elseif (in_array($this->vehicleType, [VehicleType::MOTOR, VehicleType::SCOOTER])) {
            return VehicleTypeMotorOptions::getAll();
        } elseif ($this->vehicleType === VehicleType::RECREATION) {
            return VehicleTypeRecreationalOptions::getAll();
        }

        throw new Exception("Error, unsuported vehicleType '{$this->vehicleType}'");
    }

    public function getMostFrequent(): array
    {
        return self::MOST_FREQUENT[$this->vehicleType];
    }

    public function getForCompanyVehicle(): array
    {
        return $this->companyVehicle ? self::COMPANY_VEHICLE : [];
    }

    public function getForBrand(): array
    {
        return self::BRAND_SPECIFIC[$this->brand] ?? [];
    }
}
