<?php

namespace Dealerdirect\PhoneNumber;

use Dealerdirect\PhoneNumber\Enum\CountryCode;
use Dealerdirect\PhoneNumber\Enum\CountryName;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumber as LibPhoneNumber;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;

/**
 * Wrapper for the "giggsey/libphonenumber-for-php" packages.
 *
 * The main goal of this class is to be given phone-numbers as provided by
 * various Dealerdirect databases (often stored as BIGINT, INT or VARCHAR) and
 * return them in the to E.164 format.
 *
 * In order to make sense of the mess our phone-numbers are often in, an
 * indication of which country a phone-number is located in MUST be provided.
 *
 * A convenience method is provided for the main functionality this class provides.
 *
 * E.164 is an convention that defines worldwide telephone numbers. Numbers are
 * limited to a maximum of 15 digits, excluding the international call prefix.
 * The presentation of a number is prefixed with the plus sign "+", indicating
 * that the number includes the country calling code. Please not that (as always)
 * there ARE exceptions to the 15 digit limit, valid numbers in Germany have been
 * assigned that are longer than this.
 *
 * For ful details see item 15 of:
 *
 *      https://github.com/googlei18n/libphonenumber/blob/master/FALSEHOODS.md
 */
class PhoneNumber
{
    ////////////////////////////// CLASS PROPERTIES \\\\\\\\\\\\\\\\\\\\\\\\\\\\

    const ERROR_COULD_NOT_PARSE = 'Could not parse phone number "%s"';
    const ERROR_INVALID_COUNTRY_ID_TYPE = 'Country ID must be numeric, "%s" given';
    const ERROR_INVALID_PHONE_NUMBER = 'Given string "%s" is not a valid phone number';

    /** @var static */
    private static $instance;

    /** @var PhoneNumberUtil */
    private $phoneNumberUtil;

    //////////////////////////// SETTERS AND GETTERS \\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * Returns a singleton of this class, creating one if it does not (yet) exist.
     *
     * @return static
     */
    private static function getInstance()
    {
        if (static::$instance === null) {
            $phoneNumberUtil = PhoneNumberUtil::getInstance();

            $instance = new static($phoneNumberUtil);

            static::$instance = $instance;
        }

        return static::$instance;
    }

    /**
     * @return int
     */
    private function getNumberFormat()
    {
        return PhoneNumberFormat::E164;
    }

    //////////////////////////////// PUBLIC API \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @param PhoneNumberUtil $phoneNumberUtil
     */
    final public function __construct(PhoneNumberUtil $phoneNumberUtil)
    {
        $this->phoneNumberUtil = $phoneNumberUtil;
    }

    /**
     * Converts a given phone-number for a given country to the E.164 format.
     *
     * @param int $countryId
     * @param string|int $phoneNumber
     *
     * @return string
     *
     * @throws \InvalidArgumentException
     */
    final public static function from($countryId, $phoneNumber)
    {
        $self = static::getInstance();

        if (is_numeric($countryId) === false) {
            throw $self->createException(self::ERROR_INVALID_COUNTRY_ID_TYPE, [gettype($countryId)]);
        }

        $countryCode = $self->getCountryCodeForCountryId((int) $countryId);

        return $self->format($countryCode, $phoneNumber);
    }

    /**
     * @param string $countryCode
     * @param string|int $phoneNumber
     *
     * @return string
     *
     * @throws \InvalidArgumentException
     */
    final public function format($countryCode, $phoneNumber)
    {
        // @NOTE: This also converts integers to string
        $phoneNumber = trim($phoneNumber);

        $parsedPhone = $this->parse($countryCode, $phoneNumber);

        if ($parsedPhone === null && ctype_digit($phoneNumber) === true && $phoneNumber{0} !== 0) {
            /* @NOTE: Try again with a plus-sign '+' for MySQL data-type INT or BIGINT
             *        (all-digit phone-numbers without leading zero) */
            $parsedPhone = $this->parse($countryCode, '+' . $phoneNumber);
        }

        if ($parsedPhone === null) {
            throw $this->createException(self::ERROR_INVALID_PHONE_NUMBER, [$phoneNumber]);
        }

        return $this->phoneNumberUtil->format($parsedPhone, $this->getNumberFormat());
    }

    ////////////////////////////// UTILITY METHODS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @param string $error
     * @param array $params
     *
     * @return \InvalidArgumentException
     */
    private function createException($error, array $params = [])
    {
        return new \InvalidArgumentException(vsprintf($error, $params));
    }

    private function getCountryCodeForCountryId($countryId)
    {
        /** @CHECKME: Should the default be NL or "Unknown Country code" ? */
        $countryCode = CountryName::NETHERLANDS;

        $key = CountryCode::getKey($countryId);

        if ($key !== false) {
            $countryCode = constant(CountryName::class . '::' . $key);
        }

        return $countryCode;
    }

    /**
     * @param string $countryCode
     * @param string $phoneNumber
     *
     * @return LibPhoneNumber|null
     *
     * @throws \InvalidArgumentException
     */
    private function parse($countryCode, $phoneNumber)
    {
        try {
            $parsedPhone = $this->phoneNumberUtil->parse($phoneNumber, $countryCode, null, true);
        } catch (NumberParseException $parseException) {
            throw $this->createException(self::ERROR_COULD_NOT_PARSE, [$phoneNumber]);
        }

        $isValid = $this->phoneNumberUtil->isValidNumber($parsedPhone);

        if ($isValid === false) {
            $parsedPhone = null;
        }

        return $parsedPhone;
    }
}

/*EOF*/
