<?php

namespace Dealerdirect\Pricing\Providers;

use DateTime;
use DealerDirect\Generic\Category\Fuel;
use Dealerdirect\Pricing\Vehicle;
use Dealerdirect\Generic\Category\VehicleType;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class BidHistoryProvider extends Provider
{
    public function getRequirements(): array
    {
        return [
            Vehicle::BRAND_ID => ['required', 'integer'],
            Vehicle::MODEL_ID => ['required', 'integer'],
            Vehicle::CONSTRUCTION_YEAR => ['required', 'integer'],
            Vehicle::FUEL => ['required', 'integer', Rule::in(Fuel::getAll())],
            Vehicle::ENGINE_CAPACITY => ['required', 'integer'],
            Vehicle::POWER => ['required', 'integer'],
            Vehicle::TYPE => ['required', 'string', Rule::in([
                VehicleType::CAR, VehicleType::CAMPER, VehicleType::CARAVAN,
            ])]
        ];
    }

    public function __invoke(Vehicle $vehicle): array
    {
        $this->vehicle = $vehicle;
        $sensibleBids = $this->getSensibleBidsByCollection($this->fetch(), 'bid');

        $data = [
            'average' => (int) $sensibleBids->average('bid'),
            'min' => (int) $sensibleBids->min('bid'), // Lowest bid found in bidhistory
            'max' => (int) $sensibleBids->max('bid'), // Highest bid found in bidhistory
            'num' => (int) $sensibleBids->count(), // Number of total bids found
            // Perhaps add some data concerning age of bids
        ];

        return $data;
    }

    public function getName(): string
    {
        return 'DealerdirectBidHistory';
    }

    private function fetch(): Collection
    {
        $vehicle = $this->vehicle;
        $currentYear = (int) (new DateTime())->format('Y');

        $results = DB::select(self::QUERY, [
            'brandId' => $vehicle->get(Vehicle::BRAND_ID),
            'modelId' => $vehicle->get(Vehicle::MODEL_ID),
            'fuelType' => $vehicle->get(Vehicle::FUEL),
            'ageFrom' => $currentYear - $vehicle->get(Vehicle::CONSTRUCTION_YEAR),
            'ageTill' => $currentYear - $vehicle->get(Vehicle::CONSTRUCTION_YEAR),
            'power' => $vehicle->get(Vehicle::POWER),
            'engineCapacity' => $vehicle->get(Vehicle::ENGINE_CAPACITY),
        ]);

        return collect($results);
    }

    public function isEnabled(): bool
    {
        return true;
    }

    public function isReadOnly(): bool
    {
        return true;
    }

    private function aggregator(array $data): int
    {
        if (!empty($data)) {
            $sum = array_reduce($data, fn ($carry, $row) => $carry + $row->bid, 0);
            $num = count($data);

            $result = $sum / $num;
        }

        return $result ?? 0;
    }

    // TODO: Make this query also available for motorbikes
    private const QUERY = <<<SQL
        SELECT
            last_bid_biedingen_base.bod AS bid,
            last_bid_biedingen_base.hbieder AS dealer_id,
            last_bid_biedingen_base.datum AS bid_date,
            vb.id AS lot_id
        FROM
            dealer01_ddmain.verzamel_base AS vb
            LEFT JOIN (
                SELECT
                    t1.*
                FROM
                    dealer01_ddmain.biedingen_base AS t1
                    LEFT JOIN
                        dealer01_ddmain.biedingen_base AS t2
                        ON t1.kavel = t2.kavel
                        AND t1.status = t2.status
                        AND (
                            t1.datum < t2.datum
                            OR (
                                t1.datum = t2.datum
                                AND t1.tijd < t2.tijd
                            )
                        )
                WHERE
                    t2.kavel IS NULL
                    AND t1.status = 'active'
                    AND YEAR(t1.datum) > (YEAR(NOW()) - 1)
                ) AS last_bid_biedingen_base
                ON last_bid_biedingen_base.kavel = vb.id
            LEFT JOIN
                dealer01_ddmain.bellijst_sort AS bl
                ON bl.kavelref = vb.id
        WHERE
            vb.automerk = :brandId
            AND vb.autotype = :modelId
            AND vb.brandstof = :fuelType
            AND vb.motorinhoud = :engineCapacity
            AND vb.vermogen = :power
            AND vb.reference = 1
            AND vb.bouwjaar
                BETWEEN (YEAR(NOW()) - ((:ageFrom + 1) + YEAR(NOW()) - YEAR(vb.datumtijd)))
                AND (YEAR(NOW()) - ((:ageTill - 1) + YEAR(NOW()) - YEAR(vb.datumtijd)))
            AND bl.status != 21
        ORDER BY last_bid_biedingen_base.bod DESC
        LIMIT 50
        SQL;
}
