<?php

namespace Dealerdirect\Pricing\Providers;

use DealerDirect\Generic\Category\Fuel;
use Dealerdirect\Generic\Category\VehicleType;
use Dealerdirect\Pricing\Providers\Provider;
use Dealerdirect\Pricing\Vehicle;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\Rule;

class DataWorks4YouProvider extends Provider
{
    // TODO: Get API_KEY from config
    private const API_KEY = null;

    private const BASE_URL = 'http://api.dataworks4you.nl/api/1';

    private const HEADERS = [
        'API-Key' => self::API_KEY
    ];

    public function __invoke(Vehicle $vehicle): array
    {
        $this->vehicle = $vehicle;
        $url = $this->buildUrl('request');
        $response = Http::withHeaders(self::HEADERS)->get($url);

        $xml = simplexml_load_string($response->body());
        $pricing = collect();
        foreach ($xml->children() as $child) {
            $childData = get_object_vars($child);
            if (!empty($childData)) {
                $pricing->add((int) $childData['price']);
            }
        }

        $sensibleBids = $this->getSensibleBidsByCollection($pricing);

        return [
            'average' => (int) $sensibleBids->average(),
            'min' => (int) $sensibleBids->min(), // Lowest bid found in bidhistory
            'max' => (int) $sensibleBids->max(), // Highest bid found in bidhistory
            'num' => (int) $sensibleBids->count(), // Number of total bids found
            // Perhaps add some data concerning age of bids
        ];
    }

    public function getName(): string
    {
        return 'DataWorks4You';
    }

    public function isEnabled(): bool
    {
        return true;
    }

    public function isReadOnly(): bool
    {
        return true;
    }

    private function mapper(Vehicle $vehicle): array
    {
        return [
            'reference' => 'testing', // Needed for DataWorks4You
            'brand' => $this->mapBrandName($vehicle->get(Vehicle::BRAND)),
            'model' => $this->mapModelName($vehicle->get(Vehicle::MODEL)),
            'first_reg' => $vehicle->get(Vehicle::CONSTRUCTION_YEAR),
            'engine_power' => $vehicle->get(Vehicle::POWER),
            'fuel' => $this->mapFuelId($vehicle->get(Vehicle::FUEL)),
        ];
    }

    private function mapFuelId(int $fuelId)
    {
        $mapping = [
            // Fuel::UNKNOWN => 0,
            Fuel::GASOLINE => 'B',
            Fuel::DIESEL => 'D',
            Fuel::ELECTRICITY => 'E',
            Fuel::LPG => 'G',
            Fuel::LPG_G3 => 'G',
            // Fuel::CRYOGENIC => 6,
            // Fuel::CNG => 7,
            // Fuel::HYDROGEN => 8,
            Fuel::HYBRID_GASOLINE => 'PHB',
            Fuel::HYBRID_DIESEL => 'PHD',
            Fuel::HYBRID_GASOLINE_LPG => 'PHB',

            // RESPONSE (source):
            // "Aardgas" => 898203
            // "Benzine" => 898200
            // "Diesel" => 898205
            // "Elektrisch" => 898206
            // "Ethanol" => 1640030
            // "LPG" => 898207
            // "Hybride" => 898209
            // "Overig" => 898848
            // "Plug-in Hybride Benzine" => 898215
            // "Plug-in Hybride Diesel" => 898216
            // "Super" => 898211
            // "Waterstof" => 898212
        ];

        return $mapping[$fuelId] ?? null;
    }

    private function mapBrandName(string $brandName)
    {
        return $brandName;
    }

    private function mapModelName(string $modelName)
    {
        return ucwords(strtolower(str_replace('-', ' ', $modelName)));
    }

    public function getRequirements(): array
    {
        return [
            Vehicle::TYPE => ['required', 'string', Rule::in(VehicleType::getAll())],
            Vehicle::BRAND => ['required', 'string'],
            Vehicle::MODEL => ['required', 'string'],
            Vehicle::CONSTRUCTION_YEAR => ['required', 'integer'],
            Vehicle::POWER => ['required', 'integer'],
            Vehicle::FUEL => ['required', 'integer'],
        ];
    }

    private function buildUrl(string $url)
    {
        $params = [];
        foreach ($this->mapper($this->vehicle) as $key => $value) {
            if (!empty($value)) {
                $params[] = "$key=$value";
            }
        }

        return sprintf('%s/%s?%s', self::BASE_URL, trim($url, '/'), implode('&', $params));
    }

    // private function getReferencesFromSource(string $referenceName, array $params = [], Closure $callback = null)
    // {
    //     $cacheName = $referenceName . implode('-', $params);
    //     if (!Cache::has($cacheName)) {
    //         $response = Http::withHeaders(self::HEADERS)->get(self::BASE_URL . "/reference/$referenceName");
    //         $xml = simplexml_load_string($response->body());
    //         $results = [];
    //         foreach ($xml->children() as $child) {
    //             $childData = get_object_vars($child);
    //             if ($callback) {
    //                 // Only add to $results if callback (check) is true
    //                 if ($callback($childData) === true) {
    //                     $results[$childData['name']] = (int) $childData['id'];
    //                 }
    //                 continue;
    //             }

    //             $results[$childData['name']] = (int) $childData['id'];
    //         }


    //         Cache::add($cacheName, $results);
    //     }

    //     return Cache::get($cacheName);
    // }
}
