<?php

namespace Dealerdirect\Pricing\Providers;

use Dealerdirect\Pricing\Vehicle;
use Illuminate\Support\Collection;

abstract class Provider
{
    abstract public function __invoke(Vehicle $vehicle): array;

    /**
     * Returns minimal requirements for Vehicle to be able to run current Provider
     * Note: Only "required" attributes should be in here, no "nullable" attributes as it's already done in Vehicle
     *
     * @return array
     */
    abstract public function getRequirements(): array;

    abstract public function getName(): string;

    abstract public function isEnabled(): bool;

    abstract public function isReadOnly(): bool;

    abstract public function formatRawData(array $rawData, array $translations): array;

    protected function getSensibleBidsByCollection(Collection $bids, string $column = null)
    {
        $sensibleBids = collect();
        $bids = $bids->filter(function ($bid) use ($column) {
            $bid = (int) ($column ? $bid->$column : $bid);

            return !($bid < 10);
        });

        $num = $bids->count();
        if ($num !== 0) {
            $variance = 0.0;
            $average = $bids->average($column);

            foreach ($bids as $bid) {
                $bid = $column ? $bid->$column : $bid;
                $variance += pow(($bid - $average), 2);
            }

            $standardDeviation = (float) sqrt($variance / $num);
            $sensibleBids = $bids->filter(function ($value) use ($standardDeviation, $average, $column) {
                $value = $column ? $value->$column : $value;

                return abs($average - $value) <= 2 * $standardDeviation;
            });
        }

        return $sensibleBids;
    }
}
