<?php

namespace Dealerdirect\Pricing\Tests\Feature;

use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\Pricing\Providers\BidHistoryProvider;
use Dealerdirect\Pricing\Providers\DataWorks4YouProvider;
use Dealerdirect\Pricing\Service;
use Dealerdirect\Pricing\Tests\TestCase;
use Dealerdirect\Pricing\Vehicle;
use Illuminate\Database\ConnectionInterface;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

/**
 * @covers Service
 */
class PricingTest extends TestCase
{
    use WithFaker;

    /**
     * @dataProvider provideValidVehiclesForGetBidHistory
     */
    public function testGetBidHistory(Vehicle $vehicle)
    {
        $bidData = $this->getBidHistoryData();
        $connectionInterface = $this->createStub(ConnectionInterface::class);
        $connectionInterface->expects($this->once())->method('select')->willReturn($bidData);

        DB::shouldReceive('connection')->andReturn($connectionInterface);

        $service = new Service([BidHistoryProvider::class]);
        $actual = $service->run($vehicle);

        $this->assertArrayHasKey('DealerdirectBidHistory', $actual);
        $this->assertArrayHasKey('average', $actual['DealerdirectBidHistory']);
        $this->assertArrayHasKey('min', $actual['DealerdirectBidHistory']);
        $this->assertArrayHasKey('max', $actual['DealerdirectBidHistory']);
        $this->assertArrayHasKey('num', $actual['DealerdirectBidHistory']);
    }

    /**
     * @dataProvider provideValidVehiclesForDataWorks4You
     */
    public function testDataWorks4You(Vehicle $vehicle)
    {
        $vehicleType = $vehicle->get(Vehicle::TYPE);
        Http::fake([
            'http://api.dataworks4you.nl/api/1/request*' => Http::response(
                file_get_contents(dirname(__DIR__) . "/includes/dw4y-{$vehicleType}.xml")
            )
        ]);

        $service = new Service([DataWorks4YouProvider::class]);
        $actual = $service->run($vehicle);

        $this->assertArrayHasKey('DataWorks4You', $actual);
        $this->assertArrayHasKey('average', $actual['DataWorks4You']);
        $this->assertArrayHasKey('min', $actual['DataWorks4You']);
        $this->assertArrayHasKey('max', $actual['DataWorks4You']);
        $this->assertArrayHasKey('num', $actual['DataWorks4You']);
    }

    public function provideValidVehiclesForDataWorks4You()
    {
        return [
            'valid motorbike' => [
                new Vehicle([
                    Vehicle::TYPE => VehicleType::MOTOR,
                    Vehicle::BRAND => 'KTM', // Would be nice to select brand from generic makenModel lists
                    // Vehicle::MODEL => '1290 SUPER DUKE R',
                    Vehicle::MODEL => 'SUPERDUKE 1290 R',
                    // Kawasaki ZX-6R
                    // Suzuki GSX-R600
                    Vehicle::POWER => 73,
                    Vehicle::FUEL => 1,
                    Vehicle::CONSTRUCTION_YEAR => 2014,
                ])
            ],
            'valid car' => [
                new Vehicle([
                    Vehicle::TYPE => VehicleType::CAR,
                    Vehicle::BRAND => 'BMW',
                    Vehicle::MODEL => '3-SERIE',
                    Vehicle::POWER => 135,
                    Vehicle::FUEL => 1,
                    Vehicle::CONSTRUCTION_YEAR => 2019,
                ])
            ],
        ];
    }

    public function provideValidVehiclesForGetBidHistory()
    {
        return [
            'valid motorbike' => [
                new Vehicle([
                    Vehicle::TYPE => VehicleType::MOTOR,
                    Vehicle::BRAND_ID => 169, // Would be nice to select brand from generic makenModel lists
                    Vehicle::MODEL_ID => 19928, // Would be nice to select model from generic makenModel lists
                    Vehicle::POWER => 55,
                    Vehicle::FUEL => 1,
                    Vehicle::CONSTRUCTION_YEAR => 2015,
                ])
            ],
            'valid car' => [
                new Vehicle([
                    Vehicle::TYPE => VehicleType::CAR,
                    Vehicle::BRAND_ID => 23,
                    Vehicle::MODEL_ID => 101,
                    Vehicle::POWER => 135,
                    Vehicle::FUEL => 1,
                    Vehicle::CONSTRUCTION_YEAR => 2019,
                    Vehicle::ENGINE_CAPACITY => 1998,
                ])
            ],
        ];
    }

    private function getBidHistoryData(int $num = null): array
    {
        $num = random_int(4, 50);
        $data = [];
        for ($i = 0; $i < $num; $i++) {
            $data[] = (object) [
                'bid' => $this->faker->randomNumber(5),
                'dealer_id' => $this->faker->randomNumber(6),
                'bid_date' => $this->faker->dateTimeBetween('- 1 year')->format('Y-m-d H:i:s'),
                'lot_id' => $this->faker->randomNumber(6),
            ];
        }

        return $data;
    }
}
