<?php

namespace Dealerdirect\Pricing\Tests\Unit;

use Dealerdirect\Pricing\Providers\BidHistoryProvider;
use Dealerdirect\Pricing\Providers\DataWorks4YouProvider;
use Dealerdirect\Pricing\Providers\ExampleProvider;
use Dealerdirect\Pricing\Service;
use Dealerdirect\Pricing\Vehicle;
use Dealerdirect\Pricing\Tests\TestCase;
use Dealerdirect\Generic\Category\VehicleType;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

/**
 * @covers Service
 */
class PricingServiceTest extends TestCase
{
    use WithFaker;

    public function testServiceShouldThrowExceptionWhenGivenProviderIsInvalid()
    {
        $this->expectException(Exception::class);

        $providers = [BidHistoryProvider::class, DateTime::class];
        $vehicle = new Vehicle([
            Vehicle::TYPE => VehicleType::CAR,
        ]);

        $service = new Service($providers);
        $service->run($vehicle);
    }

    /**
     * @dataProvider provideVehicleData
     */
    public function testRunService(Vehicle $vehicle, array $providers)
    {
        $expectedKeys = [];
        foreach ($providers as &$provider) {
            $providerName = $provider->getName();
            $expectedKeys[] = $providerName;
            $provider = $this->createStub(get_class($provider));
            $provider->method('getName')->willReturn($providerName);
            $provider->expects($this->once())->method('__invoke')->willReturn([
                'average' => $this->faker->randomNumber(5),
                'min' => $this->faker->randomNumber(5),
                'max' => $this->faker->randomNumber(5),
                'num' => $this->faker->numberBetween(1, 50),
            ]);
        }

        $service = new Service($providers);
        $actual = $service->run($vehicle);

        $this->assertNotEmpty($actual);

        foreach ($expectedKeys as $key) {
            $this->assertArrayHasKey($key, $actual);
        }
    }

    public function testServiceWithEmptyVehicle()
    {
        $service = new Service();
        $actual = $service->run(new Vehicle([]));

        $this->assertEmpty($actual);
    }

    public function testSingleProvider()
    {
        $vehicle = new Vehicle([
            Vehicle::TYPE => VehicleType::CAR,
            // Vehicle::BRAND_ID => 23,
            // Vehicle::MODEL_ID => 101,
            Vehicle::CONSTRUCTION_YEAR => 2019,
            Vehicle::FUEL => 1,
            Vehicle::ENGINE_CAPACITY => 1998,
            Vehicle::POWER => 135,
            // Vehicle::LICENSE_PLATE => '12AB34',
            Vehicle::BRAND => 'BMW',
            Vehicle::MODEL => '3-SERIE',
        ]);

        // TODO: Mockprovider instead of real one?
        $providerName = (new DataWorks4YouProvider())->getName();
        $provider = $this->createStub(DataWorks4YouProvider::class);
        $provider->method('getName')->willReturn($providerName);

        $service = new Service([$provider]);

        $actual = $service->run($vehicle);

        // Returned array has provided data
        $this->assertNotEmpty($actual);
        // Provider name is one of the provided entries
        $this->assertArrayHasKey($provider->getName(), $actual);
    }

    public function testOutputForDataworks4U()
    {
        $this->markTestSkipped("Don't bother testing providers in service test");

        Http::fake([
            'http://api.dataworks4you.nl/api/1/request*' => Http::response(
                file_get_contents(dirname(__DIR__) . '/includes/vehicles.xml')
            )
        ]);

        DB::shouldReceive('select')
            ->once()
            ->andReturn(
                array_fill(0, 10, (object) [
                    'bid' => $this->faker->randomNumber(5),
                    'dealer_id' => $this->faker->randomNumber(6),
                    'bid_date' => $this->faker->dateTimeBetween('- 1 year'),
                    'lot_id' => $this->faker->randomNumber(6),
                ])
            );

        $vehicle = new Vehicle([
            Vehicle::TYPE => VehicleType::CAR,
            Vehicle::BRAND_ID => 23,
            Vehicle::BRAND => 'BMW',
            Vehicle::MODEL_ID => 101,
            Vehicle::MODEL => '3-SERIE',
            Vehicle::CONSTRUCTION_YEAR => 2019,
            Vehicle::FUEL => 1,
            Vehicle::ENGINE_CAPACITY => 1998,
            Vehicle::POWER => 135,
            Vehicle::LICENSE_PLATE => '12AB34',
        ]);

        $service = new Service();
        $service->run($vehicle);

        Http::assertSent(function (Request $request) {
            return $request->url() === 'http://api.dataworks4you.nl/api/1/request?reference=testing'
                . '&brand=BMW&model=3%20Serie&first_reg=2019&engine_power=135&fuel=B';
        });
    }

    public function provideVehicleData()
    {
        return [
            'valid for all providers' => [
                new Vehicle([
                    Vehicle::TYPE => VehicleType::CAR,
                    Vehicle::BRAND_ID => 23,
                    Vehicle::BRAND => 'BMW',
                    Vehicle::MODEL_ID => 101,
                    Vehicle::MODEL => '3-SERIE',
                    Vehicle::CONSTRUCTION_YEAR => 2019,
                    Vehicle::FUEL => 1,
                    Vehicle::ENGINE_CAPACITY => 1998,
                    Vehicle::POWER => 135,
                    Vehicle::LICENSE_PLATE => '12AB34'
                ]),
                [new ExampleProvider(), new BidHistoryProvider(), new DataWorks4YouProvider()],
            ],
            'valid for ExampleProvider' => [
                new Vehicle([
                    Vehicle::TYPE => VehicleType::CAR,
                    Vehicle::LICENSE_PLATE => '12AB34',
                ]),
                [new ExampleProvider()],
            ],
        ];
    }
}
