<?php

namespace Dealerdirect\Pricing\Providers;

use DateTime;
use DealerDirect\Generic\Category\Fuel;
use DealerDirect\Generic\Category\Site;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Category\VehicleTypeCarTransmission;
use Dealerdirect\Pricing\Helpers\UriTemplate;
use Dealerdirect\Pricing\Providers\Provider;
use Dealerdirect\Pricing\Vehicle;
use Dealerdirect\Generic\Enums;
use Exception;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\Rule;

class IndicataProvider extends Provider
{
    private const BASE_URL = 'https://ws.indicata.com/vivi/v2';

    private $client;

    private $template;

    public function __construct()
    {
        $this->template = new UriTemplate();
    }

    public function __invoke(Vehicle $vehicle): array
    {
        $link = $this->getLink($this->welcome(), 'identification');
        $uri = $this->template->expand($link['href'], ['assumption' => 'FULL']);
        $response = $this->getClient()->withBody($this->createCsv($vehicle), 'text/csv')->post($uri);

        $identification = $this->poll('identification', $response);
        $report = $this->getReport($identification);

        // For now get the first as we know there is only 1 vehicle in the report
        $report = reset($report);
        $content = $report['content'];
        $valuations = [];
        $prices = collect();
        foreach ($content as $reportData) {
            if (!array_key_exists('identification', $reportData)) {
                continue;
            }

            $vehicleIdentification = $reportData['identification'];
            if (!array_key_exists('valuation', $vehicleIdentification)) {
                continue;
            }

            foreach ($vehicleIdentification['valuation'] as $valuation) {
                if (!array_key_exists('href', $valuation)) {
                    continue;
                }

                $uri = $this->template->expand($valuation['href'], [
                    'profiles' => [
                        'SUPPLY_DEMAND',
                        'RETAIL_100',
                        'MAX_PURCHASE_PRICE_100',
                        // 'PDF', // The PDF profile is only available in the Sequence API and not in the Bulk API.
                        'COMPETITIVE_SET',
                        // 'FORECAST', // Gives error: No access to profile(s): Forecast
                    ],
                    // Not sure to add the odometer, as it's already in the identification
                    'odometer' => $vehicle->get(Vehicle::ODOMETER),
                ]);
                $valuationResponse = $this->getClient()->get($uri);

                $valuationResult = $valuationResponse->json();

                $valuations[] = $valuationResult;
                $this->addPriceByValuation($valuationResult, $prices);
            }
        }

        return [
            'average' => $prices->average(),
            'min' => $prices->min(), // Lowest bid found in bidhistory
            'max' => $prices->max(), // Highest bid found in bidhistory
            'num' => $prices->count(), // Number of total bids found
            'raw' => $valuations, // Raw valutation data for pricing details
            // Perhaps add some data concerning age of bids
        ];
    }

    public function getName(): string
    {
        return 'Indicata';
    }

    public function getRequirements(): array
    {
        // TODO: Set all requirements
        return [
            Vehicle::CLASSIFICATION => ['required', Rule::in([
                Enums\Classification::YOUNG,
                Enums\Classification::EXPORT,
                Enums\Classification::OTHER,
            ])],
            Vehicle::REFERENCE => ['required', Rule::in([
                Site::REF_CAR_NL,
                Site::REF_CAR_BE_NL,
                Site::REF_CAR_MARKTPLAATS_NL,
            ])],
            Vehicle::MODEL_ID => ['required', 'integer', 'gt:0'],
            Vehicle::TYPE => ['required', 'string', Rule::in([
                VehicleType::CAR,
                VehicleType::CAMPER,
                VehicleType::CARAVAN,
            ])],
        ];
    }

    public function isEnabled(): bool
    {
        return true;
    }

    public function isReadOnly(): bool
    {
        return true;
    }

    public function formatRawData(array $rawData, array $translations): array
    {
        if (!array_key_exists('competitiveSet', $rawData)) {
            // Double check if it's not in the first row of $rawData
            $rawData = reset($rawData);
            if (!array_key_exists('competitiveSet', $rawData)) {
                return [];
            }
        }

        $formattedData = [];
        foreach ($rawData['competitiveSet'] as $competitiveSet) {
            $formattedData['data'][] = [
                "make" => $rawData['make']['name'],
                "model" => $rawData['model']['name'],
                "registrationDate" => $this->formatIndicataRegDate($competitiveSet['regDate']['name']),
                "facelift" => $competitiveSet['facelift']['name'],
                "engineData" => $this->formatIndicataEngine($competitiveSet['engine']['name']),
                "bodyType" => $competitiveSet['body']['name'],
                "odometerData" => [
                    "value" => $this->formatOdometerValue(
                        $competitiveSet['odometer']['distance'],
                        $competitiveSet['odometer']['unit']
                    ),
                    "difference" => $competitiveSet['odometer']['difference'],
                ],
                "priceData" => [
                    "value" => $this->formatCorrectedPriceValue(
                        $competitiveSet['correctedPrice']['amount']['value'],
                        $competitiveSet['correctedPrice']['amount']['currency']
                    ),
                    "difference" => $competitiveSet['correctedPrice']['difference'],
                ],
            ];
        }

        foreach (reset($formattedData['data']) as $key => $value) {
            $formattedData['headers'][$key] = [
                "name" => $key,
                "span" => is_array($value) ? count($value) : 1,
            ];
        }

        return $formattedData;
    }

    private function getClient()
    {
        if (is_null($this->client)) {
            $this->client = Http::withBasicAuth(
                config('pricing.indicata.username'),
                config('pricing.indicata.password')
            )->withHeaders([
                'Accept-Encoding' => 'gzip',
                'Accept-Language' => 'en_GB',
                'Accept' => 'application/json',
            ]);
        }

        return $this->client;
    }

    private function mapFuel(int $fuelId): string
    {
        $fuelTypes = [
            FUEL::UNKNOWN => '',
            FUEL::GASOLINE => 'PETROL',
            FUEL::DIESEL => 'DIESEL',
            FUEL::ELECTRICITY => 'Electricity',
            FUEL::LPG => 'LPG',
            FUEL::LPG_G3 => 'LPG',
            FUEL::CRYOGENIC => 'CRYOGENIC',
            FUEL::CNG => 'CNC',
            FUEL::HYDROGEN => 'HYDROGEN',
            FUEL::HYBRID_GASOLINE => 'PETROL',
            FUEL::HYBRID_DIESEL => 'DIESEL',
            FUEL::HYBRID_GASOLINE_LPG => 'PETROL',
        ];

        return $fuelTypes[$fuelId] ?? '';
    }

    private function mapTransmission(int $transmissionId): string
    {
        $transmissionTypes = [
            VehicleTypeCarTransmission::UNKNOWN => '',
            VehicleTypeCarTransmission::MANUAL => 'Manual',
            VehicleTypeCarTransmission::AUTOMATIC => 'Automatic',
            VehicleTypeCarTransmission::SEMI_AUTOMATIC => 'Semi Automatic',
        ];

        return $transmissionTypes[$transmissionId] ?? '';
    }

    private function createCsv(Vehicle $vehicle)
    {
        $power = $vehicle->get(Vehicle::POWER);
        $values = [
            '#' => 1,
            'country' => $vehicle->get(Vehicle::COUNTRY_CODE, ''),
            'category' => '',
            'make' => $vehicle->get(Vehicle::BRAND, ''),
            'model' => $vehicle->get(Vehicle::MODEL, ''),
            'variant' => $vehicle->get(Vehicle::TRIM, ''),
            'firstreg' => $vehicle->has(Vehicle::FIRST_REGISTRATION_DATE)
                ? (DateTime::createFromFormat('Y-m-d', $vehicle->get(Vehicle::FIRST_REGISTRATION_DATE)))->format('Y-m')
                : $vehicle->get(Vehicle::CONSTRUCTION_YEAR, ''),
            'body' => '',
            'doors' => $vehicle->get(Vehicle::NUMBER_OF_DOORS, ''),
            'engine' => !empty($power) ? "{$power}PK" : '',
            'engineliters' => $vehicle->get(Vehicle::ENGINE_CAPACITY, '') . 'ccm',
            'fuel' => $this->mapFuel($vehicle->get(Vehicle::FUEL, '')),
            'transmission' => $this->mapTransmission($vehicle->get(Vehicle::TRANSMISSION, '')),
            'wheeldrive' => $vehicle->get(Vehicle::WHEEL_DRIVE, ''),
            'seats' => $vehicle->get(Vehicle::NUMBER_OF_SEATS, ''),
            'trim' => $vehicle->get(Vehicle::TRIM, ''),
        ];

        $csv = implode(',', array_keys($values)) . PHP_EOL;
        $csv .= implode(',', $values);

        return $csv;
    }

    private function welcome()
    {
        $uri = self::BASE_URL . '/bulk';
        $result = $this->getClient()->get($uri);
        if (!$result->successful()) {
            throw new Exception(
                "Error fetching Indicata. Got code: '{$result->status()}' with reason: '{$result->reason()}'"
            );
        }

        return $result->json();
    }

    private function getLink(array $resource, string $relation)
    {
        $filtered = array_values(array_filter($resource['links'] ?? [], function ($l) use ($relation) {
            return $l['rel'] === $relation;
        }));

        return $filtered && count($filtered) > 0 ? $filtered[0] : null;
    }

    private function addPriceByValuation(array $valuationData, Collection &$prices)
    {
        foreach ($valuationData['valuations'] ?? [] as $valuation) {
            $amount = $valuation['amount'];
            if ($amount['currency'] !== 'EUR') {
                throw new Exception(
                    "Error, could not handle other currencies than EUR. Given currency {$amount['currency']}"
                );
            }

            $prices->add((int) $amount['value']);
        }

        return $prices;
    }

    private function getReport(array $completed)
    {
        $result = [];
        $link = $this->getLink($completed, 'report');
        $page = null;
        $num = 1;

        do {
            $uri = $this->template->expand($link['href'], []);
            $res = $this->getClient()->get($uri);
            $page = $res->json();
            $result[] = $page;
            $num = $num + 1;
        } while (
            array_key_exists('links', $page)
            && $page['links'] !== null
            && ($link = $this->getLink($page, 'next'))
        );

        return $result;
    }

    private function poll(string $type, Response $response): array
    {
        $response->header('X-Bulk-Job-Id');
        $stat = $response->json();

        while (true) {
            sleep(config('pricing.indicata.poll-sleep-time', 5));

            $uri = $this->getLink($stat, 'status')['href'];
            $statusResponse = $this->getClient()->get($uri);

            $stat = $statusResponse->json();
            if (in_array($stat['status'], ['QUEUED', 'PROCESSING', 'COMPILING'])) {
                throw new \Exception("Unhandled status {$stat['status']}");
            } elseif ($stat['status'] === 'ERROR') {
                throw new \Exception($stat['status']);
            }

            return $stat;
        }

        return $stat;
    }

    private function formatIndicataRegDate($registrationDate): string
    {
        $formattedRegDate = 'unknown';
        $pattern = '/^\d{4}/';

        $result = preg_match($pattern, $registrationDate, $matches);

        if ($result > 0) {
            $formattedRegDate = $matches[0];
        }

        return $formattedRegDate;
    }

    private function formatIndicataEngine($engineText): string
    {
        $formattedEngineText = 'unknown';
        $pattern = '/^.+ \d+kW/';

        $result = preg_match($pattern, $engineText, $matches);

        if ($result > 0) {
            $formattedEngineText = $matches[0];
        }

        return $formattedEngineText;
    }

    private function formatOdometerValue($value, $unit): string
    {
        return "$unit$value";
    }

    private function formatCorrectedPriceValue($value, $unit): string
    {
        return "$unit$value";
    }
}
