<?php

namespace Dealerdirect\Pricing;

use DealerDirect\Generic\Category\Fuel;
use DealerDirect\Generic\Category\Site;
use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\Generic\Enums;
use Exception;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class Vehicle
{
    public const BODY_TYPE = 'bodyType';
    public const BRAND = 'brand';
    public const BRAND_ID = 'brandId';
    public const CALLGROUP_ID = 'callgroupId';
    public const CLASSIFICATION = 'classification';
    public const CONSTRUCTION_YEAR = 'constructionYear';
    public const COUNTRY_CODE = 'countryCode';
    public const ENGINE_CAPACITY = 'engineCapacity';
    public const FIRST_REGISTRATION_DATE = 'firstRegistrationDate';
    public const FUEL = 'fuelType';
    public const LICENSE_PLATE = 'licensePlate';
    public const MODEL = 'model';
    public const MODEL_ID = 'modelId';
    public const NUMBER_OF_DOORS = 'numberOfDoors';
    public const NUMBER_OF_SEATS = 'numberOfSeats';
    public const ODOMETER = 'odometer';
    public const POWER = 'power';
    public const TRANSMISSION = 'transmission';
    public const TRIM = 'trim';
    public const TYPE = 'vehicleType';
    public const WHEEL_DRIVE = 'wheelDrive';
    public const UPHOLSTERY = 'upholstery';
    public const OPTIONS_ACCESORIES = 'optionsAccesories';
    public const VALUE_ADDED_TAX = 'valueAddedTax';
    public const REFERENCE = 'reference';

    private array $data = [];

    /**
     * @param array $data
     * @throws Exception
     */
    public function __construct(array $data)
    {
        $this->data = $this->validate($data);
    }

    public function get($key, $default = null): mixed
    {
        return $this->data[$key] ?? $default;
    }

    public function has(string $key): bool
    {
        return array_key_exists($key, $this->data) && !is_null($this->data[$key]);
    }

    public function toArray(): array
    {
        return $this->data;
    }

    /**
     * Validates data. All validation rules have to be nullable here.
     * This is only to make sure all the data is valid and in all the providers the same.
     *
     * @param array $data
     * @return array
     * @throws Exception
     */
    private function validate(array $data): array
    {
        try {
            return Validator::validate($data, [
                self::BODY_TYPE => ['nullable', 'integer'],
                self::BRAND => ['nullable', 'string'],
                self::BRAND_ID => ['nullable', 'integer'],
                self::CALLGROUP_ID => ['nullable', 'integer'],
                self::CLASSIFICATION => ['nullable', Rule::enum(Enums\Classification::class)],
                self::CONSTRUCTION_YEAR => ['nullable', 'date_format:Y'],
                self::COUNTRY_CODE => ['nullable', Rule::in('NL', 'BE', 'DE')],
                self::ENGINE_CAPACITY => ['nullable'],
                self::FIRST_REGISTRATION_DATE => ['nullable', 'date_format:Y-m-d'],
                self::FUEL => ['nullable', 'integer', Rule::in(Fuel::getAll())],
                self::LICENSE_PLATE => ['nullable', 'string'],
                self::MODEL => ['nullable', 'string'],
                self::MODEL_ID => ['nullable', 'integer'],
                self::NUMBER_OF_DOORS => ['nullable', 'integer'],
                self::NUMBER_OF_SEATS => ['nullable', 'integer'],
                self::ODOMETER => ['nullable', 'integer'],
                self::POWER => ['nullable'],
                self::TRANSMISSION => ['nullable', 'integer'],
                self::TRIM => ['nullable', 'string'],
                self::TYPE => ['nullable', 'string', Rule::in(VehicleType::getAll())],
                self::WHEEL_DRIVE => ['nullable', 'nullable', 'string'],
                self::UPHOLSTERY => ['nullable', 'integer'],
                self::OPTIONS_ACCESORIES => ['nullable', 'array'],
                self::REFERENCE => ['nullable', Rule::in(Site::all())],
            ]);
        } catch (ValidationException $exception) {
            // TODO: Create our own exceptions
            throw new Exception("Error, invalid data in pricing Vehicle", 0, $exception);
        }
    }
}
