<?php

namespace Dealerdirect\Pricing\Tests\Feature;

use DealerDirect\Generic\Category\Upholstery;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Category\VehicleTypeCarOptions;
use DealerDirect\Generic\Category\VehicleTypeCarTransmission;
use Dealerdirect\Pricing\Providers\BidHistoryProvider;
use Dealerdirect\Pricing\Providers\DataWorks4YouProvider;
use Dealerdirect\Pricing\Providers\IndicataProvider;
use Dealerdirect\Pricing\Providers\JPCarsProvider;
use Dealerdirect\Pricing\Service;
use Dealerdirect\Pricing\Tests\TestCase;
use Dealerdirect\Pricing\Vehicle;
use Illuminate\Database\ConnectionInterface;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

/**
 * @covers Service
 */
class PricingTest extends TestCase
{
    use WithFaker;

    public function testJPCars()
    {
        Http::fake([
            'https://api.jp.cars/api/valuate/extended?formatter=en' => Http::response(
                file_get_contents(dirname(__DIR__) . "/includes/JPCars/valuate.json")
            ),
            'https://api.jp.cars/api/headers/specials' => Http::response(
                file_get_contents(dirname(__DIR__) . "/includes/JPCars/specials.json")
            ),
        ]);
        $vehicle = new Vehicle([
            Vehicle::CALLGROUP_ID => $this->faker->randomElement([1, 11, 8, 18, 21, 968]),
            Vehicle::COUNTRY_CODE => 'NL',
            Vehicle::LICENSE_PLATE => '23XXD2',
            Vehicle::ODOMETER => 100000,
            Vehicle::TYPE => VehicleType::CAR,
            Vehicle::BRAND => 'VOLKSWAGEN',
            Vehicle::TRIM => 'Move Up!',
            Vehicle::UPHOLSTERY => Upholstery::LEATHER,
            Vehicle::OPTIONS_ACCESORIES => [
                VehicleTypeCarOptions::ALLOY_WHEEL_RIM_18_INCH,
                VehicleTypeCarOptions::AIRCO,
                VehicleTypeCarOptions::NAVIGATION_SYSTEM,
                VehicleTypeCarOptions::XENON_LIGHTS,
            ],
        ]);

        $service = new Service([JPCarsProvider::class]);
        $actual = $service->run($vehicle);

        $this->assertArrayHasKey('JPCars', $actual);
        $this->assertArrayHasKey('average', $actual['JPCars']);
        $this->assertArrayHasKey('min', $actual['JPCars']);
        $this->assertArrayHasKey('max', $actual['JPCars']);
        $this->assertArrayHasKey('num', $actual['JPCars']);
    }

    /**
     * @dataProvider provideValidVehiclesForGetBidHistory
     */
    public function testGetBidHistory(array $vehicleData)
    {
        $vehicle = new Vehicle($vehicleData);
        $bidData = $this->getBidHistoryData();
        $connectionInterface = $this->createStub(ConnectionInterface::class);
        $connectionInterface->expects($this->once())->method('select')->willReturn($bidData);

        DB::shouldReceive('connection')->andReturn($connectionInterface);

        $service = new Service([BidHistoryProvider::class]);
        $actual = $service->run($vehicle);

        $this->assertArrayHasKey('DealerdirectBidHistory', $actual);
        $this->assertArrayHasKey('average', $actual['DealerdirectBidHistory']);
        $this->assertArrayHasKey('min', $actual['DealerdirectBidHistory']);
        $this->assertArrayHasKey('max', $actual['DealerdirectBidHistory']);
        $this->assertArrayHasKey('num', $actual['DealerdirectBidHistory']);
    }

    /**
     * @dataProvider provideValidVehiclesForDataWorks4You
     */
    public function testDataWorks4You(array $vehicleData)
    {
        $vehicle = new Vehicle($vehicleData);
        $vehicleType = $vehicle->get(Vehicle::TYPE);
        Http::fake([
            'http://api.dataworks4you.nl/api/1/request*' => Http::response(
                file_get_contents(dirname(__DIR__) . "/includes/dw4y-{$vehicleType}.xml")
            )
        ]);

        $service = new Service([DataWorks4YouProvider::class]);
        $actual = $service->run($vehicle);

        $this->assertEmpty($actual, 'DataWorks4YouProvider is disabled');
    }

    public function testIndicata()
    {
        Http::fake([
            'https://ws.indicata.com/vivi/v2/bulk' => Http::response(
                file_get_contents(dirname(__DIR__) . "/includes/indicata/welcome.json")
            ),

            'https://ws.indicata.com/vivi/v2/bulk/identification?assumption=FULL' => Http::response(
                file_get_contents(dirname(__DIR__) . "/includes/indicata/identification.json")
            ),

            'https://ws.indicata.com/vivi/v2/bulk/identification/9fc11737-d56b-4e13-adf5-295546cd0c25/status' =>
                Http::response(
                    file_get_contents(dirname(__DIR__) . "/includes/indicata/status.json")
                ),

            'https://ws.indicata.com/vivi/v2/bulk/identification/9fc11737-d56b-4e13-adf5-295546cd0c25/report' =>
                Http::response(
                    file_get_contents(dirname(__DIR__) . "/includes/indicata/report-1.json")
                ),

            'https://ws.indicata.com/vivi/v2/'
            . 'MGhmTmlKeXNCalRQNHhNajVLcU1XWnYvYkc0MjlKSFo0enFEeG9OUlJmNGp4MDZFQjQ1Ti9NMWFEb0hzeG5ldQ'
            . '==/NL/497520:81244:5/valuation/RETAIL_100,SUPPLY_DEMAND?regdate=2019&odometer=34000' =>
                Http::response(
                    file_get_contents(dirname(__DIR__) . "/includes/indicata/valuation.json")
                ),
        ]);

        $vehicle = new Vehicle([
            Vehicle::CALLGROUP_ID => 1,
            Vehicle::TYPE => VehicleType::CAR,
            Vehicle::COUNTRY_CODE => 'NL',
            Vehicle::BRAND => 'BMW',
            Vehicle::MODEL => '3-SERIE',
            Vehicle::TRIM => '318i M Sport Corporate Lease',
            Vehicle::ODOMETER => 34000,
            Vehicle::CONSTRUCTION_YEAR => 2019,
            Vehicle::NUMBER_OF_DOORS => 5,
            Vehicle::BODY_TYPE => '',
            Vehicle::ENGINE_CAPACITY => 1998,
            Vehicle::FUEL => 1,
            Vehicle::TRANSMISSION => VehicleTypeCarTransmission::AUTOMATIC,
            Vehicle::WHEEL_DRIVE => 'RWD',
            Vehicle::NUMBER_OF_SEATS => 5,
        ]);

        $service = new Service([IndicataProvider::class]);
        $actual = $service->run($vehicle);

        $this->assertArrayHasKey('Indicata', $actual);
        $this->assertArrayHasKey('average', $actual['Indicata']);
        $this->assertArrayHasKey('min', $actual['Indicata']);
        $this->assertArrayHasKey('max', $actual['Indicata']);
        $this->assertArrayHasKey('num', $actual['Indicata']);
    }

    public function provideValidVehiclesForDataWorks4You()
    {
        return [
            'valid motorbike' => [
                [
                    Vehicle::TYPE => VehicleType::MOTOR,
                    Vehicle::BRAND => 'KTM', // Would be nice to select brand from generic makenModel lists
                    // Vehicle::MODEL => '1290 SUPER DUKE R',
                    Vehicle::MODEL => 'SUPERDUKE 1290 R',
                    // Kawasaki ZX-6R
                    // Suzuki GSX-R600
                    Vehicle::POWER => 73,
                    Vehicle::FUEL => 1,
                    Vehicle::CONSTRUCTION_YEAR => 2014,
                ]
            ],
            'valid car' => [
                [
                    Vehicle::TYPE => VehicleType::CAR,
                    Vehicle::BRAND => 'BMW',
                    Vehicle::MODEL => '3-SERIE',
                    Vehicle::POWER => 135,
                    Vehicle::FUEL => 1,
                    Vehicle::CONSTRUCTION_YEAR => 2019,
                ]
            ],
        ];
    }

    public function provideValidVehiclesForGetBidHistory()
    {
        return [
            'valid motorbike' => [
                [
                    Vehicle::TYPE => VehicleType::MOTOR,
                    Vehicle::BRAND_ID => 169, // Would be nice to select brand from generic makenModel lists
                    Vehicle::MODEL_ID => 19928, // Would be nice to select model from generic makenModel lists
                    Vehicle::POWER => 55,
                    Vehicle::FUEL => 1,
                    Vehicle::CONSTRUCTION_YEAR => 2015,
                ]
            ],
            'valid car' => [
                [
                    Vehicle::TYPE => VehicleType::CAR,
                    Vehicle::BRAND_ID => 23,
                    Vehicle::MODEL_ID => 101,
                    Vehicle::POWER => 135,
                    Vehicle::FUEL => 1,
                    Vehicle::CONSTRUCTION_YEAR => 2019,
                    Vehicle::ENGINE_CAPACITY => 1998,
                ]
            ],
        ];
    }

    private function getBidHistoryData(int $num = null): array
    {
        $num = random_int(4, 50);
        $data = [];
        for ($i = 0; $i < $num; $i++) {
            $data[] = (object) [
                'bid' => $this->faker->randomNumber(5),
                'dealer_id' => $this->faker->randomNumber(6),
                'bid_date' => $this->faker->dateTimeBetween('- 1 year')->format('Y-m-d H:i:s'),
                'lot_id' => $this->faker->randomNumber(6),
                'odometer' => $this->faker->randomNumber(6),
            ];
        }

        return $data;
    }
}
