<?php

namespace Dealerdirect\Pricing\Tests\Unit;

use Dealerdirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Category\VehicleTypeCarBodyType;
use DealerDirect\Generic\Category\VehicleTypeCarOptions;
use Dealerdirect\Pricing\Providers\BidHistoryProvider;
use Dealerdirect\Pricing\Providers\DataWorks4YouProvider;
use Dealerdirect\Pricing\Providers\ExampleProvider;
use Dealerdirect\Pricing\Providers\IndicataProvider;
use Dealerdirect\Pricing\Providers\JPCarsProvider;
use Dealerdirect\Pricing\Service;
use Dealerdirect\Pricing\Tests\TestCase;
use Dealerdirect\Pricing\Vehicle;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;

/**
 * @covers Service
 */
class PricingServiceTest extends TestCase
{
    use WithFaker;

    public function testServiceShouldThrowExceptionWhenGivenProviderIsInvalid()
    {
        $this->expectException(Exception::class);

        $providers = [BidHistoryProvider::class, DateTime::class];
        $vehicle = new Vehicle([
            Vehicle::TYPE => VehicleType::CAR,
        ]);

        $service = new Service($providers);
        $service->run($vehicle);
    }

    /**
     * @dataProvider provideVehicleData
     */
    public function testRunService(array $vehicleData, array $providers)
    {
        $vehicle = new Vehicle($vehicleData);
        $expectedKeys = [];
        foreach ($providers as &$provider) {
            $providerName = $provider->getName();
            $isEnabled = $provider->isEnabled();
            $provider = $this->createPartialMock(get_class($provider), ['__invoke']);

            if ($isEnabled) {
                $expectedKeys[] = $providerName;
                $provider->expects($this->once())->method('__invoke')->willReturn([
                    'average' => $this->faker->randomNumber(5),
                    'min' => $this->faker->randomNumber(5),
                    'max' => $this->faker->randomNumber(5),
                    'num' => $this->faker->numberBetween(1, 50),
                ]);
            }
        }

        $service = new Service($providers);
        $actual = $service->run($vehicle);

        $this->assertNotEmpty($actual);

        foreach ($expectedKeys as $key) {
            $this->assertArrayHasKey($key, $actual);
        }
    }

    public function testServiceWithEmptyVehicle()
    {
        $service = new Service();
        $actual = $service->run(new Vehicle([]));

        $this->assertEmpty($actual);
    }

    public function testSingleProvider()
    {
        $vehicle = new Vehicle([
            Vehicle::TYPE => VehicleType::CAR,
            // Vehicle::BRAND_ID => 23,
            // Vehicle::MODEL_ID => 101,
            Vehicle::CONSTRUCTION_YEAR => 2019,
            Vehicle::FUEL => 1,
            Vehicle::ENGINE_CAPACITY => 1998,
            Vehicle::POWER => 135,
            // Vehicle::LICENSE_PLATE => '12AB34',
            Vehicle::BRAND => 'BMW',
            Vehicle::MODEL => '3-SERIE',
        ]);

        // TODO: Mockprovider instead of real one?
        $providerName = (new IndicataProvider())->getName();
        $provider = $this->createStub(IndicataProvider::class);
        $provider->method('isEnabled')->willReturn(true);
        $provider->method('getName')->willReturn($providerName);

        $service = new Service([$provider]);

        $actual = $service->run($vehicle);

        // Returned array has provided data
        $this->assertNotEmpty($actual);
        // Provider name is one of the provided entries
        $this->assertArrayHasKey($provider->getName(), $actual);
    }

    public function provideVehicleData()
    {
        return [
            'valid for all providers' => [
                [
                    Vehicle::TYPE => VehicleType::CAR,
                    Vehicle::BRAND_ID => 23,
                    Vehicle::BRAND => 'BMW',
                    Vehicle::MODEL_ID => 101,
                    Vehicle::MODEL => '3-SERIE',
                    Vehicle::CONSTRUCTION_YEAR => 2019,
                    Vehicle::FUEL => 1,
                    Vehicle::ENGINE_CAPACITY => 1998,
                    Vehicle::POWER => 135,
                    Vehicle::LICENSE_PLATE => '12AB34',
                    Vehicle::CALLGROUP_ID => 1,
                    Vehicle::COUNTRY_CODE => 'NL',
                    Vehicle::ODOMETER => 100000,
                    Vehicle::OPTIONS_ACCESORIES => VehicleTypeCarOptions::getAll(),
                    Vehicle::BODY_TYPE => VehicleTypeCarBodyType::CONVERTIBLE,
                ],
                [
                    new BidHistoryProvider(),
                    new DataWorks4YouProvider(),
                    new ExampleProvider(),
                    new IndicataProvider(),
                    new JPCarsProvider(),
                ],
            ],
            'valid for IndicataProvider' => [
                [
                    Vehicle::TYPE => VehicleType::CAR,
                    Vehicle::LICENSE_PLATE => '12AB34',
                    Vehicle::CALLGROUP_ID => 1,
                ],
                [new IndicataProvider()],
            ],
        ];
    }
}
