<?php

namespace Dealerdirect\Pricing\Providers;

use DealerDirect\Generic\Category\Upholstery;
use Dealerdirect\Generic\Category\VehicleType;
use Dealerdirect\Pricing\Providers\Provider;
use Dealerdirect\Pricing\Vehicle;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class JPCarsProvider extends Provider
{
    private const BASE_URL = 'https://api.jp.cars/api';

    public function getName(): string
    {
        return 'JPCars';
    }

    public function getRequirements(): array
    {
        return [
            Vehicle::CALLGROUP_ID => ['required', 'integer', Rule::in([1, 11, 8, 18, 21, 968])],
            Vehicle::COUNTRY_CODE => ['required', Rule::in(['NL'])],
            Vehicle::LICENSE_PLATE => ['required', 'string'],
            Vehicle::ODOMETER => ['required', 'integer'],
            Vehicle::TYPE => ['required', 'string', Rule::in([
                VehicleType::CAR, VehicleType::CAMPER, VehicleType::CARAVAN
            ])],
        ];
    }

    public function __invoke(Vehicle $vehicle): array
    {
        $data = [
            'license_plate' => $vehicle->get(Vehicle::LICENSE_PLATE),
            'mileage' => $vehicle->get(Vehicle::ODOMETER),
        ];

        $booleanChecks = [
            Vehicle::AIRCO => 'equipment_airco',
            Vehicle::ALLOY_WHEELS => 'equipment_alloy_wheels',
            Vehicle::NAVIGATION => 'equipment_navigation',
            Vehicle::XENON_LIGHTS => 'equipment_xenon',
        ];
        foreach ($booleanChecks as $vehicleKey => $dataKey) {
            if ($vehicle->has($vehicleKey)) {
                $data[$dataKey] = $vehicle->get($vehicleKey);
            }
        }

        if ($vehicle->has(Vehicle::TRIM) && $vehicle->has(Vehicle::BRAND)) {
            $special = $this->getSpecialsByBrandAndTrim($vehicle->get(Vehicle::BRAND), $vehicle->get(Vehicle::TRIM));
            if (!empty($special)) {
                $data['specials_info'] = $special;
            }
        }

        if ($vehicle->has(Vehicle::UPHOLSTERY)) {
            $data['equipment_leather'] = in_array($vehicle->get(Vehicle::UPHOLSTERY, 0), [
                Upholstery::LEATHER, Upholstery::SEMI_LEATHER
            ]);
        }

        $response = $this->getClient()->post('/valuate', $data);
        if (!$response->successful()) {
            throw new \Exception(sprintf(
                'Request failed for JPCars pricing. Response code: "%d", Reason: "%s"',
                $response->status(),
                $response->reason(),
            ));
        }
        $data = $response->json();
        $value = $data['topdown_value'] ?? $data['value'];

        return [
            'average' => $value,
            'min' => $value,
            'max' => $value,
            'num' => $data['window_size'],
            'raw' => $data,
        ];
    }

    private function getSpecialsByBrandAndTrim(string $brand, string $trim): string
    {
        $cacheName = 'JPCars-specials';
        $specials = Cache::get($cacheName, function () use ($cacheName) {
            $response = $this->getClient()->get('/headers/specials');
            if (!$response->successful()) {
                return [];
            }

            $data = $response->json('results');
            Cache::add($cacheName, $data, 60 * 60 * 24);

            return $data;
        });

        $brand = strtolower($brand);
        $specialsToCheck = $specials['all'] ?? [];
        if (array_key_exists($brand, $specials)) {
            $specialsToCheck = $specials[$brand];
        }

        foreach ($specialsToCheck as $special) {
            if (Str::contains($trim, $special)) {
                return $special;
            }
        }

        return '';
    }

    private function getClient()
    {
        $token = config('pricing.JPCars.token');
        if (empty($token)) {
            throw new Exception('Error, No token for JPCars');
        }

        return Http::withToken($token)->baseUrl(self::BASE_URL);
    }

    public function isEnabled(): bool
    {
        return true;
    }

    public function isReadOnly(): bool
    {
        return true;
    }

    public function formatRawData(array $rawData, array $translations): array
    {
        return $rawData;
    }
}
