<?php

namespace Dealerdirect\Pricing\Providers;

use DealerDirect\Generic\Category\Fuel;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Category\VehicleTypeCarTransmission;
use Dealerdirect\Pricing\Helpers\UriTemplate;
use Dealerdirect\Pricing\Providers\Provider;
use Dealerdirect\Pricing\Vehicle;
use Exception;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\Rule;

class IndicataProvider extends Provider
{
    private const BASE_URL = 'https://ws.indicata.com/vivi/v2';

    private $client;

    private $identity;

    private $template;

    public function __construct()
    {
        $this->client = Http::withBasicAuth(config('pricing.indicata.username'), config('pricing.indicata.password'))
            ->withHeaders([
                'Accept-Encoding' => 'gzip',
                'Accept-Language' => 'en_GB',
                'Accept' => 'application/json',
            ]);

        $this->template = new UriTemplate();
    }

    public function getName(): string
    {
        return 'Indicata';
    }

    public function getRequirements(): array
    {
        // TODO: Set all requirements
        return [
            Vehicle::CALLGROUP_ID => ['required', 'integer', Rule::in([1, 11])],
            Vehicle::TYPE => ['required', 'string', Rule::in([
                VehicleType::CAR,
                VehicleType::CAMPER,
                VehicleType::CARAVAN,
            ])],
        ];
    }

    private function mapFuel(int $fuelId): string
    {
        $fuelTypes = [
            FUEL::UNKNOWN => '',
            FUEL::GASOLINE => 'PETROL',
            FUEL::DIESEL => 'DIESEL',
            FUEL::ELECTRICITY => 'Electricity',
            FUEL::LPG => 'LPG',
            FUEL::LPG_G3 => 'LPG',
            FUEL::CRYOGENIC => 'CRYOGENIC',
            FUEL::CNG => 'CNC',
            FUEL::HYDROGEN => 'HYDROGEN',
            FUEL::HYBRID_GASOLINE => 'PETROL',
            FUEL::HYBRID_DIESEL => 'DIESEL',
            FUEL::HYBRID_GASOLINE_LPG => 'PETROL',
        ];

        return $fuelTypes[$fuelId] ?? '';
    }

    private function mapTransmission(int $transmissionId): string
    {
        $transmissionTypes = [
            VehicleTypeCarTransmission::UNKNOWN => '',
            VehicleTypeCarTransmission::MANUAL => 'Manual',
            VehicleTypeCarTransmission::AUTOMATIC => 'Automatic',
            VehicleTypeCarTransmission::SEMI_AUTOMATIC => 'Semi Automatic',
        ];

        return $transmissionTypes[$transmissionId] ?? '';
    }

    private function createCsv(Vehicle $vehicle)
    {
        $values = [
            '#' => 1,
            'country' => $vehicle->get(Vehicle::COUNTRY_CODE, ''),
            'category' => '',
            'make' => $vehicle->get(Vehicle::BRAND, ''),
            'model' => $vehicle->get(Vehicle::MODEL, ''),
            'variant' => $vehicle->get(Vehicle::TRIM, ''),
            'firstreg' => $vehicle->get(Vehicle::CONSTRUCTION_YEAR, ''),
            'body' => '',
            'doors' => $vehicle->get(Vehicle::NUMBER_OF_DOORS, ''),
            'engine' => '',
            'engineliters' => $vehicle->get(Vehicle::ENGINE_CAPACITY, '') . 'ccm',
            'fuel' => $this->mapFuel($vehicle->get(Vehicle::FUEL, '')),
            'transmission' => $this->mapTransmission($vehicle->get(Vehicle::TRANSMISSION, '')),
            'wheeldrive' => $vehicle->get(Vehicle::WHEEL_DRIVE, ''),
            'seats' => $vehicle->get(Vehicle::NUMBER_OF_SEATS, ''),
            'trim' => $vehicle->get(Vehicle::TRIM, ''),
        ];

        $csv = implode(',', array_keys($values)) . PHP_EOL;
        $csv .= implode(',', $values);

        return $csv;
    }

    private function welcome()
    {
        $uri = self::BASE_URL . '/bulk';
        $result = $this->client->get($uri);


        return $result->json();
    }

    private function getLink(array $resource, string $relation)
    {
        $filtered = array_values(array_filter($resource['links'], function ($l) use ($relation) {
            return $l['rel'] === $relation;
        }));

        return $filtered && count($filtered) > 0 ? $filtered[0] : null;
    }

    public function __invoke(Vehicle $vehicle): array
    {
        $link = $this->getLink($this->welcome(), 'identification');
        $uri = $this->template->expand($link['href'], ['assumption' => 'FULL']);
        $response = $this->client->withBody($this->createCsv($vehicle), 'text/csv')->post($uri);


        $identification = $this->poll('identification', $response);
        $report = $this->getReport($identification);

        // For now get the first as we know there is only 1 vehicle in the report
        $report = reset($report);
        $content = $report['content'];
        $valuations = [];
        $prices = collect();
        foreach ($content as $reportData) {
            if (!array_key_exists('identification', $reportData)) {
                continue;
            }

            $vehicleIdentification = $reportData['identification'];
            if (!array_key_exists('valuation', $vehicleIdentification)) {
                continue;
            }

            foreach ($vehicleIdentification['valuation'] as $valuation) {
                if (!array_key_exists('href', $valuation)) {
                    continue;
                }

                $uri = $this->template->expand($valuation['href'], [
                    'profiles' => [
                        'SUPPLY_DEMAND',
                        'RETAIL_100',
                        'MAX_PURCHASE_PRICE_100',
                        // 'PDF', // The PDF profile is only available in the Sequence API and not in the Bulk API.
                        'COMPETITIVE_SET',
                        // 'FORECAST', // Gives error: No access to profile(s): Forecast
                    ],
                    // Not sure to add the odometer, as it's already in the identification
                    'odometer' => $vehicle->get(Vehicle::ODOMETER),
                ]);
                $valuationResponse = $this->client->get($uri);


                $valuationResult = $valuationResponse->json();

                $valuations[] = $valuationResult;
                $this->addPriceByValuation($valuationResult, $prices);
            }
        }

        return [
            'average' => $prices->average(),
            'min' => $prices->min(), // Lowest bid found in bidhistory
            'max' => $prices->max(), // Highest bid found in bidhistory
            'num' => $prices->count(), // Number of total bids found
            // Perhaps add some data concerning age of bids
        ];
    }

    private function addPriceByValuation(array $valuationData, Collection &$prices)
    {
        foreach ($valuationData['valuations'] ?? [] as $valuation) {
            $amount = $valuation['amount'];
            if ($amount['currency'] !== 'EUR') {
                throw new Exception(
                    "Error, could not handle other currencies than EUR. Given currency {$amount['currency']}"
                );
            }

            $prices->add((int) $amount['value']);
        }

        return $prices;
    }

    private function getReport(array $completed)
    {
        $result = [];
        $link = $this->getLink($completed, 'report');
        $page = null;
        $num = 1;

        do {
            $uri = $this->template->expand($link['href'], []);
            $res = $this->client->get($uri);
            $page = $res->json();
            $result[] = $page;
            $num = $num + 1;
        } while (
            array_key_exists('links', $page)
            && $page['links'] !== null
            && ($link = $this->getLink($page, 'next'))
        );

        return $result;
    }

    private function poll(string $type, Response $response): array
    {
        $this->identity = $response->header('X-Bulk-Job-Id');
        $stat = $response->json();

        while (true) {
            sleep(config('pricing.indicata.poll-sleep-time', 5));

            $uri = $this->getLink($stat, 'status')['href'];
            $statusResponse = $this->client->get($uri);


            $stat = $statusResponse->json();
            if (in_array($stat['status'], ['QUEUED', 'PROCESSING', 'COMPILING'])) {
                throw new \Exception("Unhandled status {$stat['status']}");
            } elseif ($stat['status'] === 'ERROR') {
                throw new \Exception($stat['status']);
            }

            return $stat;
        }

        return $stat;
    }

    public function isEnabled(): bool
    {
        return true;
    }

    public function isReadOnly(): bool
    {
        return true;
    }
}
