<?php

namespace Dealerdirect\Pricing\Providers;

use DealerDirect\Generic\Category\Upholstery;
use Dealerdirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Category\VehicleTypeCarBodyType;
use DealerDirect\Generic\Category\VehicleTypeCarOptions;
use Dealerdirect\Pricing\Providers\Provider;
use Dealerdirect\Pricing\Vehicle;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class JPCarsProvider extends Provider
{
    private const BASE_URL = 'https://api.jp.cars/api';

    public function getName(): string
    {
        return 'JPCars';
    }

    public function isEnabled(): bool
    {
        return true;
    }

    public function isReadOnly(): bool
    {
        return true;
    }

    public function getRequirements(): array
    {
        return [
            Vehicle::CALLGROUP_ID => ['required', 'integer', Rule::in([
                1, // Jong & Fris 1 (NL)
                7, // Export 1 (NL)
                8, // Overig 1 (NL)
                11, // Jong & Fris (BE)
                17, // Export (BE)
                18, // Overig 1 (BE)
                19, // Export 2 (NL)
                21, // Overig 2 (NL)
                968 // Marktplaats (NL)
            ])],
            Vehicle::COUNTRY_CODE => ['required', Rule::in(['NL'])],
            Vehicle::LICENSE_PLATE => ['required', 'string'],
            Vehicle::ODOMETER => ['required', 'integer'],
            Vehicle::TYPE => ['required', 'string', Rule::in([
                VehicleType::CAR, VehicleType::CAMPER, VehicleType::CARAVAN
            ])],
        ];
    }

    public function __invoke(Vehicle $vehicle): array
    {
        $equipments = $this->getEquipments($vehicle);
        $data = [
            'license_plate' => $vehicle->get(Vehicle::LICENSE_PLATE),
            'mileage' => $vehicle->get(Vehicle::ODOMETER),
            'equipment' => implode(',', $equipments),
            'specials_info' => implode(',', $this->getSpecials($vehicle, $equipments)),
        ];

        $queryParams = http_build_query(['formatter' => 'en']);
        $response = $this->getClient()->post("/valuate?$queryParams", $data);
        if (!$response->successful()) {
            throw new \Exception(sprintf(
                'Request failed for JPCars pricing. Response code: "%d", Reason: "%s"',
                $response->status(),
                $response->reason(),
            ));
        }
        $data = $response->json();
        $value = $data['topdown_value'] ?? $data['value'];

        return [
            'average' => $value,
            'min' => $value,
            'max' => $value,
            'num' => $data['window_size'],
            'raw' => $data,
        ];
    }

    public function formatRawData(array $rawData, array $translations): array
    {
        return $rawData;
    }

    private function getAvailableSpecials()
    {
        $cacheName = 'JPCars-specials';

        return Cache::get($cacheName, function () use ($cacheName) {
            $response = $this->getClient()->get('/headers/specials');
            if (!$response->successful()) {
                return [];
            }

            $data = $response->json('results');
            Cache::add($cacheName, $data, 60 * 60 * 24);

            return $data;
        });
    }

    private function getClient()
    {
        $token = config('pricing.JPCars.token');
        if (empty($token)) {
            throw new Exception('Error, No token for JPCars');
        }

        return Http::withToken($token)->baseUrl(self::BASE_URL);
    }

    private function getSpecials(Vehicle $vehicle, array $equipments): array
    {
        $specials = [];
        $availableSpecials = $this->getAvailableSpecials();
        $specialsToCheck = $availableSpecials['all'] ?? [];
        $brand = $vehicle->get(Vehicle::BRAND);
        if (array_key_exists($brand, $availableSpecials)) {
            $specialsToCheck = array_merge($specialsToCheck, $availableSpecials[$brand]);
        }

        $trim = $vehicle->get(Vehicle::TRIM);
        if (!empty($trim)) {
            // Always add trim to $specials when it's available
            $specials[] = $trim;
            foreach ($specialsToCheck as $special) {
                if (Str::contains($trim, $special)) {
                    $specials[] = $special;
                }
            }
        }

        if ($vehicle->get(Vehicle::BODY_TYPE) === VehicleTypeCarBodyType::TRAILER) {
            $specials[] = 'oplegger';
        }

        $optionsMapping = [
            VehicleTypeCarOptions::REAR_CAMERA => 'Achteruitrijcamera',
            VehicleTypeCarOptions::ELECTRIC_TOW_BAR => 'Trekhaak',
            VehicleTypeCarOptions::DETACHABLE_TOW_BAR => 'Trekhaak',
            VehicleTypeCarOptions::MULTIFUNCTION_STEERING_WHEEL => 'MF Stuurwiel',
            VehicleTypeCarOptions::PARKING_ASSISTANCE_SYSTEM => 'PDC', // Park distance control
            VehicleTypeCarOptions::AUTOMATIC_PARKING => 'PDC', // Park distance control
            VehicleTypeCarOptions::AUTOMATIC_AIRCO => 'Climate Control'
        ];
        foreach ($vehicle->get(Vehicle::OPTIONS_ACCESORIES, []) as $option) {
            if (array_key_exists($option, $optionsMapping)) {
                $specials[] = $optionsMapping[$option];
            }
        }

        // Also add the equipments to $specials
        $specials = array_merge($specials, $equipments);

        return array_unique($specials);
    }

    private function getEquipments(Vehicle $vehicle)
    {
        $equipment = [];
        $optionsMapping = [
            VehicleTypeCarOptions::AIRCO => 'airco',
            VehicleTypeCarOptions::ABS => 'abs',
            VehicleTypeCarOptions::FOUR_WHEEL_DRIVE => '4x4',
            VehicleTypeCarOptions::AUTOMATIC_AIRCO => 'automatic_air_conditioning',
            VehicleTypeCarOptions::AIRBAG => 'airbag',
            VehicleTypeCarOptions::TUNING => 'tuning',
            VehicleTypeCarOptions::LEATHER_SEATS => 'leather',
            VehicleTypeCarOptions::AIRBAG_PASSENGER => 'passenger_airbag',
            VehicleTypeCarOptions::METALLIC_RIMS => 'alloy_wheels',
            VehicleTypeCarOptions::ELECTRIC_WINDOWS => 'electric_windows',
            VehicleTypeCarOptions::SIDE_AIRBAGS => 'side_airbags',
            VehicleTypeCarOptions::ROOF_RAILS => 'roofrails',
            VehicleTypeCarOptions::NAVIGATION_SYSTEM => 'navigation',
            VehicleTypeCarOptions::XENON_LIGHTS => 'xenon',
            VehicleTypeCarOptions::ON_BOARD_COMPUTER => 'on_board_computer',
            VehicleTypeCarOptions::ELECTRIC_ADJUSTABLE_SEATS => 'electric_adjustable_seats',
            VehicleTypeCarOptions::CENTRAL_LOCKING => 'central_door_locking',
            VehicleTypeCarOptions::PARKING_ASSISTANCE_SYSTEM => 'park_assist',
            // VehicleTypeCarOptions::OPEN_ROOF => 'Sun/Moon-roof',
            VehicleTypeCarOptions::ALARM => 'alarm',
            VehicleTypeCarOptions::FOG_LIGHTS => 'fog_lights',
            VehicleTypeCarOptions::SEAT_HEATING => 'heated_seats',
            VehicleTypeCarOptions::IMMOBILIZER => 'immobilizer',
            VehicleTypeCarOptions::POWER_STEERING => 'power_stearing',
            VehicleTypeCarOptions::CD_PLAYER => 'cd_player',
            VehicleTypeCarOptions::TRACTION_CONTROL => 'traction_control',
            VehicleTypeCarOptions::CRUISE_CONTROL => 'cruise_control',
            VehicleTypeCarOptions::RADIO => 'radio',
            VehicleTypeCarOptions::ESP => 'esp',
            VehicleTypeCarOptions::TOWING_HOOK => 'towbar',
            VehicleTypeCarOptions::PANORAMIC_ROOF => 'panoramaroof',
            VehicleTypeCarOptions::AIR_SUSPENSION => 'air_suspension',
            VehicleTypeCarOptions::START_STOP_SYSTEM => 'start_stop_system',
            VehicleTypeCarOptions::HEADLAMP_SPRAYERS => 'headlight_sprayer',
            VehicleTypeCarOptions::SPORT_CHASSIS => 'sport_suspension',
            VehicleTypeCarOptions::ARMREST_CENTER => 'armrest',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_17_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_18_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_19_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_20_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_21_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::MULTIFUNCTION_STEERING_WHEEL => 'multi_function_steering_wheel',
            VehicleTypeCarOptions::LED_HEADLIGHTS => 'led_headlights',
            VehicleTypeCarOptions::PRIVACY_GLAS => 'privacy_glass',
            VehicleTypeCarOptions::REAR_CAMERA => 'rear_camera',
            VehicleTypeCarOptions::AUTOMATIC_PARKING => 'automatic_parking',
            VehicleTypeCarOptions::LEVEL_GUIDANCE => 'level_guidance',
            VehicleTypeCarOptions::ADAPTIVE_CRUISE_CONTROL => 'adaptive_cruise_control',
            VehicleTypeCarOptions::ELECTRICALLY_ADJUSTABLE_MIRRORS => 'electrical_adjustable_mirrors',
            VehicleTypeCarOptions::LEATHER_STEERING_WHEEL => 'leather_steering_wheel',
            VehicleTypeCarOptions::LEATHER_KNOB => 'leather_shift_knob',
            VehicleTypeCarOptions::TIRE_PRESSURE_SENSORS => 'tire_pressure_system',
            VehicleTypeCarOptions::KEYLESS_ENTRY => 'keyless_entry',
            VehicleTypeCarOptions::SPORT_CHAIRS => 'sport_seats',
            VehicleTypeCarOptions::FLASHING_LIGHTS_IN_MIRRORS => 'mirror_indicators',
            VehicleTypeCarOptions::CHANGED_RADIO => 'radio',
            VehicleTypeCarOptions::SIDE_BARS => 'side_bars',
            VehicleTypeCarOptions::BULL_BAR => 'bull_bar',
            VehicleTypeCarOptions::HEAD_UP_DISPLAY => 'head_up_display',
            VehicleTypeCarOptions::VIRTUAL_COCKPIT => 'virtual_cockpit',
            VehicleTypeCarOptions::LANE_ASSIST => 'lane_assistent',
            VehicleTypeCarOptions::AUTO_PILOT => 'auto_pilot',
            VehicleTypeCarOptions::CAMERA_360_DEGREES => '360_degrees_camera',
            VehicleTypeCarOptions::ELECTRIC_TAILGATE => 'electric_tailgate',
            VehicleTypeCarOptions::ELECTRIC_TOW_BAR => 'electric_tow_bar',
            VehicleTypeCarOptions::DETACHABLE_TOW_BAR => 'detachable_tow_bar',
            VehicleTypeCarOptions::MASSAGE_SEATS => 'massage_seats',
            VehicleTypeCarOptions::SEAT_VENTILATION => 'seat_ventilation',
            VehicleTypeCarOptions::PARKING_HEATER => 'parking_heater',
            VehicleTypeCarOptions::MEMORY_SEATS => 'memory_seats',
            VehicleTypeCarOptions::PREMIUM_SOUND_SYSTEM => 'premium_sound_system',
        ];

        foreach ($vehicle->get(Vehicle::OPTIONS_ACCESORIES, []) as $option) {
            if (array_key_exists($option, $optionsMapping)) {
                $equipment[] = $optionsMapping[$option];
            }
        }

        if (in_array($vehicle->get(Vehicle::UPHOLSTERY, 0), [Upholstery::LEATHER, Upholstery::SEMI_LEATHER], true)) {
            $equipment[] = 'equipment_leather';
        }

        return array_unique($equipment);
    }
}
