<?php

namespace Dealerdirect\Pricing;

use Dealerdirect\Pricing\Providers\BidHistoryProvider;
use Dealerdirect\Pricing\Providers\DataWorks4YouProvider;
use Dealerdirect\Pricing\Providers\IndicataProvider;
use Dealerdirect\Pricing\Providers\JPCarsProvider;
use Dealerdirect\Pricing\Providers\Provider;
use Exception;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class Service
{
    public const PROVIDERS = [
        BidHistoryProvider::class,
        DataWorks4YouProvider::class,
        IndicataProvider::class,
        JPCarsProvider::class,
    ];

    // private Vehicle $vehicle;

    /** @var Provider[] */
    private array $providers = [];

    /**
     * @param array $vehicle
     */
    public function __construct(array $providers = self::PROVIDERS)
    {
        foreach ($providers as $provider) {
            $this->addProvider($provider);
        }
    }

    public function addProvider($provider)
    {
        if (!is_subclass_of($provider, Provider::class)) {
            $providerClass = Provider::class;
            $provider = is_string($provider) ? $provider : get_class($provider);

            throw new Exception("ERROR, given provider '$provider' must be an instance of '$providerClass'");
        }

        $instance = is_string($provider) ? new $provider() : $provider;
        $providerName = $instance->getName();
        if (array_key_exists($providerName, $this->providers)) {
            throw new Exception("ERROR, Provider name '{$providerName}' is not unique.");
        }

        $this->providers[$providerName] = $instance;
    }

    public function run(Vehicle $vehicle)
    {
        $results = [];
        foreach ($this->providers as $providerName => $provider) {
            // Only run providers that are enabled
            if (!$provider->isEnabled()) {
                continue;
            }

            try {
                $this->validate($vehicle->toArray(), $provider->getRequirements());
            } catch (ValidationException $exception) {
                $this->logInvalidVehicle($exception, $provider, $vehicle);
                continue;
            }

            try {
                $results[$providerName] = $provider($vehicle);
            } catch (Exception $exception) {
                logger()->error('Error while fetching price', [
                    'exceptionMessage' => $exception->getMessage(),
                    'providerClass' => get_class($provider),
                    'providerName' => $providerName,
                    'vehicle' => $vehicle->toArray(),
                ]);
                continue;
            }
        }

        return $results;
    }

    public function findProviderByName(string $name)
    {
        return $this->providers[$name] ?? null;
    }

    private function validate(array $data, array $rules): array
    {
        $validator = Validator::make($data, $rules);

        return $validator->validate();
    }

    private function logInvalidVehicle(ValidationException $exception, Provider $provider, Vehicle $vehicle)
    {
        $data = [
            'provider' => $provider,
            'vehicle' => $vehicle,
            'errors' => $exception->errors(),
        ];
        logger()->debug('vehicle not valid for provider', $data);
    }
}
