<?php

namespace Dealerdirect\Pricing\Providers;

use DealerDirect\Generic\Category\Upholstery;
use DealerDirect\Generic\Category\ValueAddedTax;
use Dealerdirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Category\VehicleTypeCarBodyType;
use DealerDirect\Generic\Category\VehicleTypeCarOptions;
use Dealerdirect\Pricing\Exceptions\JPCarsException;
use Dealerdirect\Pricing\Providers\Provider;
use Dealerdirect\Pricing\Vehicle;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class JPCarsProvider extends Provider
{
    private const BASE_URL = 'https://api.jp.cars/api';

    private const RESPONSE_ERRORS = [
        'ERROR_VALUATION_MIN_BUILD',
        'ERROR_WINDOW_EMPTY',
        'ERROR_VALUATION_MODEL',
    ];

    public function getName(): string
    {
        return 'JPCars';
    }

    public function isEnabled(): bool
    {
        return true;
    }

    public function isReadOnly(): bool
    {
        return true;
    }

    public function getRequirements(): array
    {
        return [
            Vehicle::CALLGROUP_ID => ['required', 'integer', Rule::in([
                1, // Jong & Fris 1 (NL)
                4, // Bestel en vracht (NL)
                7, // Export 1 (NL)
                8, // Overig 1 (NL)
                11, // Jong & Fris (BE)
                17, // Export (BE)
                18, // Overig 1 (BE)
                19, // Export 2 (NL)
                21, // Overig 2 (NL)
                968 // Marktplaats (NL)
            ])],
            Vehicle::COUNTRY_CODE => ['required', Rule::in(['NL'])],
            Vehicle::LICENSE_PLATE => ['required', 'string'],
            Vehicle::ODOMETER => ['required', 'integer'],
            Vehicle::TYPE => ['required', 'string', Rule::in([
                VehicleType::CAR, VehicleType::CAMPER, VehicleType::CARAVAN
            ])],
        ];
    }

    public function __invoke(Vehicle $vehicle): array
    {
        $equipments = $this->getEquipments($vehicle);
        $postData = [
            'license_plate' => $vehicle->get(Vehicle::LICENSE_PLATE),
            'mileage' => $vehicle->get(Vehicle::ODOMETER),
            'equipment' => implode(',', $equipments),
            'specials_info' => implode(',', $this->getSpecials($vehicle, $equipments)),
        ];

        $equipmentToCheck = [
            'equipment_leather' => 'equipment_leather',
            'led_headlights' => 'equipment_xenon',
            'navigation' => 'equipment_navigation',
            'four_doors' => 'four_doors',
            'xenon' => 'equipment_xenon',
            'airco' => 'equipment_airco',
            'alloy_wheels' => 'equipment_alloy_wheels',
            'open_roof' => 'equipment_roof',
            'panoramaroof' => 'equipment_roof',
        ];
        foreach ($equipmentToCheck as $checkOnKey => $equipmentKey) {
            if (in_array($checkOnKey, $equipments)) {
                $postData[$equipmentKey] = true;
            }
        }

        $queryParams = http_build_query(['formatter' => 'en']);
        $response = $this->getClient()->post("/valuate/extended?$queryParams", $postData);
        $data = $response->json();

        if (!empty($data['error'])) {
            if (!in_array($data['error'], self::RESPONSE_ERRORS, true)) {
                logger()->error('Unrecognised error in JPCars', compact('postData', 'data'));
            }

            $data['window_size'] = 0;
        }

        if (!$response->successful()) {
            throw new JPCarsException(
                $response->status(),
                $response->reason(),
                $data['error'] ?? null,
                $data['error_message'] ?? null
            );
        }
        $value = $data['topdown_value'] ?? $data['value'];

        return [
            'average' => $value,
            'min' => $value,
            'max' => $value,
            'num' => $data['window_size'],
            'raw' => $data,
        ];
    }

    public function formatRawData(array $rawData, array $translations): array
    {
        if (!array_key_exists('window', $rawData)) {
            // Double check if it's not in the first row of $rawData
            $rawData = reset($rawData);
            if (!array_key_exists('window', $rawData)) {
                return [];
            }
        }

        $formattedData = [];
        $price = $rawData['best_selling_price'] ?? $rawData['value'];
        foreach ($rawData['window'] as $similarVehicles) {
            // Set default values for the vehicle compared to
            if ($rawData['license_plate'] === $similarVehicles['license_plate']) {
                $similarVehicles['company_image_url'] = 'https://image.jp.cars/logo_dealerdirect.nl.jpg';
            }

            $formattedData['data'][] = [
                "vehicleImage" => $similarVehicles['vehicle_image_url'],
                "companyImage" => $similarVehicles['company_image_url'],
                "licensePlate" => $similarVehicles['license_plate'],
                "priceData" => [
                    "value" => sprintf("EUR%s", $similarVehicles['value']),
                    "difference" => $price - $similarVehicles['value'],
                ],
                "registrationDate" => $similarVehicles['build'],
                "daysInStock" => $similarVehicles['days_in_stock'],
                "soldSince" => $similarVehicles['sold_since'],
                "imported" => $similarVehicles['imported'],
                "odometerData" => [
                    "value" => sprintf("KILOMETER%s", $similarVehicles['mileage']),
                    "difference" => !empty($rawData['current_mileage'])
                        ? $similarVehicles['mileage'] - $rawData['current_mileage']
                        : 0,
                ],
            ];
        }

        foreach (reset($formattedData['data']) as $key => $value) {
            $formattedData['headers'][$key] = [
                "name" => $key,
                "span" => is_array($value) ? count($value) : 1,
            ];
        }

        return $formattedData;
    }

    private function getAvailableSpecials()
    {
        $cacheName = 'JPCars-specials';

        return Cache::get($cacheName, function () use ($cacheName) {
            $response = $this->getClient()->get('/headers/specials');
            if (!$response->successful()) {
                return [];
            }

            $data = $response->json('results');
            Cache::add($cacheName, $data, 60 * 60 * 24);

            return $data;
        });
    }

    private function getClient()
    {
        $token = config('pricing.JPCars.token');
        if (empty($token)) {
            throw new Exception('Error, No token for JPCars');
        }

        return Http::withToken($token)->baseUrl(self::BASE_URL);
    }

    private function getSpecials(Vehicle $vehicle, array $equipments): array
    {
        $specials = [];
        $availableSpecials = $this->getAvailableSpecials();
        $specialsToCheck = $availableSpecials['all'] ?? [];
        $brand = $vehicle->get(Vehicle::BRAND);
        if (array_key_exists($brand, $availableSpecials)) {
            $specialsToCheck = array_merge($specialsToCheck, $availableSpecials[$brand]);
        }

        $trim = $vehicle->get(Vehicle::TRIM);
        if (!empty($trim)) {
            // Always add trim to $specials when it's available
            $specials[] = $trim;
            foreach ($specialsToCheck as $special) {
                if (Str::contains($trim, $special)) {
                    $specials[] = $special;
                }
            }
        }

        if ($vehicle->get(Vehicle::BODY_TYPE) === VehicleTypeCarBodyType::TRAILER) {
            $specials[] = 'Oplegger';
        } elseif ($vehicle->get(Vehicle::BODY_TYPE) === VehicleTypeCarBodyType::PICKUP) {
            $specials[] = 'Pick up';
        }

        if ($vehicle->get(Vehicle::VALUE_ADDED_TAX) === ValueAddedTax::BTW) {
            $specials[] = 'VAT inc';
        }

        $optionsMapping = [
            VehicleTypeCarOptions::REAR_CAMERA => 'Achteruitrijcamera',
            VehicleTypeCarOptions::ELECTRIC_TOW_BAR => 'Trekhaak',
            VehicleTypeCarOptions::DETACHABLE_TOW_BAR => 'Trekhaak',
            VehicleTypeCarOptions::MULTIFUNCTION_STEERING_WHEEL => 'MF Stuurwiel',
            VehicleTypeCarOptions::PARKING_ASSISTANCE_SYSTEM => 'PDC', // Park distance control
            VehicleTypeCarOptions::AUTOMATIC_PARKING => 'PDC', // Park distance control
            VehicleTypeCarOptions::AUTOMATIC_AIRCO => 'Climate Control',
            VehicleTypeCarOptions::SEATER_7 => '7 Zitter',
            VehicleTypeCarOptions::COMPANY_CAR_PANELING => 'Betimmering',
            VehicleTypeCarOptions::COMPANY_CAR_ROOF_RACK => 'Imperiaal',
            VehicleTypeCarOptions::COMPANY_CAR_SLIDING_DOOR => 'Schuifdeur',
            VehicleTypeCarOptions::COMPANY_CAR_REFRIGERATED_TRUCK => 'Koelwagen',
            VehicleTypeCarOptions::COMPANY_CAR_BOX_TRUCK => 'Bakwagen',
            VehicleTypeCarOptions::COMPANY_CAR_DRIVE_ON_TRUCK => 'Oprijwagen',
            VehicleTypeCarOptions::COMPANY_CAR_L1 => 'L1',
            VehicleTypeCarOptions::COMPANY_CAR_L2 => 'L2',
            VehicleTypeCarOptions::COMPANY_CAR_L3 => 'L3',
            VehicleTypeCarOptions::COMPANY_CAR_L4 => 'L4',
            VehicleTypeCarOptions::COMPANY_CAR_H1 => 'H1',
            VehicleTypeCarOptions::COMPANY_CAR_H2 => 'H2',
            VehicleTypeCarOptions::COMPANY_CAR_H3 => 'H3',
            VehicleTypeCarOptions::COMPANY_CAR_H4 => 'H4',
            // Brand specific
            VehicleTypeCarOptions::LAND_ROVER_DYNAMIC => 'Dynamic',
            VehicleTypeCarOptions::MINI_JCW => 'JCW',
            VehicleTypeCarOptions::MERCEDES_AMG => 'AMG',
            VehicleTypeCarOptions::MERCEDES_WIDESCREEN => 'Widescreen',
            VehicleTypeCarOptions::AIR_SUSPENSION => 'Luchtvering',
            VehicleTypeCarOptions::CITROEN_SHINE => 'Shine',
            VehicleTypeCarOptions::CITROEN_CROSSBACK => 'Crossback',
            VehicleTypeCarOptions::FORD_ACTIVE => 'Active',
            VehicleTypeCarOptions::FORD_ST_LINE => 'ST-line',
            VehicleTypeCarOptions::FORD_S_EDITION => 'S-edition',
            VehicleTypeCarOptions::FORD_VIGNALE => 'Vignale',
            VehicleTypeCarOptions::FORD_TITANIUM => 'Titanium',
            VehicleTypeCarOptions::FORD_WINDSHIELD_HEATING => 'Voorruitverwarming',
            VehicleTypeCarOptions::OPEL_OPC => 'OPC',
            VehicleTypeCarOptions::OPEL_TWIN_TOP => 'TwinTop',
            VehicleTypeCarOptions::OPEL_GS_LINE => 'GS-Line',
            VehicleTypeCarOptions::OPEL_COSMO => 'Cosmo',
            VehicleTypeCarOptions::OPEL_INNOVATION => 'Innovation',
            VehicleTypeCarOptions::OPEL_ROCKS => 'Rocks',
            VehicleTypeCarOptions::OPEL_SPORT => 'Sport',
            VehicleTypeCarOptions::FIAT_SPORT => '500s/Sport',
            VehicleTypeCarOptions::AUDI_S_LINE => 'S-line',
            VehicleTypeCarOptions::VIRTUAL_COCKPIT => 'Virtual Cockpit',
            VehicleTypeCarOptions::BMW_GRAN_TURISMO => 'Gran Turismo',
            VehicleTypeCarOptions::BMW_M_PACKAGE => 'M pakket',
            VehicleTypeCarOptions::BMW_SPORT_SEATS => 'Sportstoelen',
            VehicleTypeCarOptions::VOLKSWAGEN_GTD => 'GTD',
            VehicleTypeCarOptions::VOLKSWAGEN_GTI => 'GTI',
            VehicleTypeCarOptions::VOLKSWAGEN_R_LINE => 'R-line',
            VehicleTypeCarOptions::VOLKSWAGEN_HIGHLINE => 'Highline',
            VehicleTypeCarOptions::VOLKSWAGEN_ALLSPACE => 'Allspace',
            VehicleTypeCarOptions::VOLKSWAGEN_ALLTRACK => 'Alltrack',
            VehicleTypeCarOptions::VOLKSWAGEN_HIGH_UP => 'High Up!',
            VehicleTypeCarOptions::VOLKSWAGEN_MOVE_UP => 'Move Up!',
            VehicleTypeCarOptions::VOLKSWAGEN_TAKE_UP => 'Take Up!',
            VehicleTypeCarOptions::VOLKSWAGEN_BLACK_STYLE => 'Black Style',
            VehicleTypeCarOptions::VOLKSWAGEN_STYLE => 'Style',
            VehicleTypeCarOptions::KIA_GT_LINE => 'GT-line',
            VehicleTypeCarOptions::KIA_PHEV => 'PHEV',
            VehicleTypeCarOptions::KIA_EXECUTIVE_LINE => 'Executive line',
            VehicleTypeCarOptions::TESLA_AUTOPILOT => 'Autopilot',
            VehicleTypeCarOptions::RENAULT_RS => 'RS',
            VehicleTypeCarOptions::RENAULT_HELLY_HANSEN => 'Helly Hansen',
            VehicleTypeCarOptions::RENAULT_GT_LINE => 'GT-line',
            VehicleTypeCarOptions::RENAULT_BOSE => 'Bose',
            VehicleTypeCarOptions::RENAULT_INTENS => 'Intens',
            VehicleTypeCarOptions::RENAULT_INITIALE => 'Initiale',
            VehicleTypeCarOptions::DACIA_STEPWAY => 'Stepway',
            VehicleTypeCarOptions::NISSAN_TEKNA => 'Tekna',
            VehicleTypeCarOptions::NISSAN_NISMO => 'Nismo',
            VehicleTypeCarOptions::NISSAN_N_DESIGN => 'N-Design',
            VehicleTypeCarOptions::NISSAN_BLACK_EDITION => 'Black Edition',
            VehicleTypeCarOptions::NISSAN_N_SPORT => 'N-Sport',
            VehicleTypeCarOptions::SEAT_FR => 'FR',
            VehicleTypeCarOptions::PEUGEOT_GT_LINE => 'GT-line',
            VehicleTypeCarOptions::PEUGEOT_ALLURE => 'Allure',
            VehicleTypeCarOptions::PEUGEOT_RXH => 'RXH',
            VehicleTypeCarOptions::PEUGEOT_TOP => 'Top',
            VehicleTypeCarOptions::SUZUKI_HIGH_EXECUTIVE => 'High Executive',
            VehicleTypeCarOptions::SUZUKI_ALL_GRIP => 'All Grip',
            VehicleTypeCarOptions::VOLVO_CROSS_COUNTRY => 'Cross Country',
            VehicleTypeCarOptions::VOLVO_SUMMUM => 'Summum',
            VehicleTypeCarOptions::VOLVO_R_DESIGN => 'R-design',
            VehicleTypeCarOptions::VOLVO_INSCRIPTION => 'Inscription',
            VehicleTypeCarOptions::PORSCHE_SPORT_CHRONO => 'Sport Chrono',
            VehicleTypeCarOptions::LEXUS_F_SPORT => 'F-sport',
            VehicleTypeCarOptions::LEXUS_PRESIDENT_LINE => 'President Line',
            VehicleTypeCarOptions::LEXUS_LUXURY_LINE => 'Luxury Line',
            VehicleTypeCarOptions::MAZDA_GT_M => 'GT-M',
            VehicleTypeCarOptions::MAZDA_SIGNATURE => 'Signature',
            VehicleTypeCarOptions::ALFA_ROMEO_TI => 'TI',
            VehicleTypeCarOptions::ALFA_ROMEO_Q4 => 'Q4',
            VehicleTypeCarOptions::HYUNDAI_I_CATCHER => 'I catcher',
            VehicleTypeCarOptions::HYUNDAI_PHEV => 'PHEV',
            VehicleTypeCarOptions::HYUNDAI_N_LINE => 'N-Line',
            VehicleTypeCarOptions::SAAB_AERO => 'Aero',
            VehicleTypeCarOptions::JEEP_SRT => 'SRT',
            VehicleTypeCarOptions::SKODA_RS => 'RS',
            VehicleTypeCarOptions::SKODA_MONTE_CARLO => 'Monte Carlo',
            VehicleTypeCarOptions::SKODA_BLACK_EDITION => 'Black Edition',
            VehicleTypeCarOptions::SKODA_SPORTLINE => 'Sportline',
            VehicleTypeCarOptions::TOYOTA_BI_TONE => 'BI-Tone',
            VehicleTypeCarOptions::TOYOTA_EXECUTIVE => 'Executive',
            VehicleTypeCarOptions::TOYOTA_X_PLAY => 'X-Play',
            VehicleTypeCarOptions::TOYOTA_X_JOY => 'X-Joy',
            VehicleTypeCarOptions::TOYOTA_GR_SPORT => 'GR Sport',
            VehicleTypeCarOptions::BUMPERS_IN_COLOR => 'Bumpers in kleur',
            VehicleTypeCarOptions::CRUISE_CONTROL => 'Cruise Control',
            VehicleTypeCarOptions::SIDE_BARS => 'Sidebars',
            VehicleTypeCarOptions::ELECTRIC_TAILGATE => 'Laadklep',
            VehicleTypeCarOptions::FOUR_WHEEL_DRIVE => '4x4',
            VehicleTypeCarOptions::REAR_CAMERA => 'Achteruitrijcamera',
            VehicleTypeCarOptions::ELECTRIC_TOW_BAR => 'Trekhaak',
            VehicleTypeCarOptions::DETACHABLE_TOW_BAR => 'Trekhaak',
            VehicleTypeCarOptions::AUTOMATIC_PARKING => 'PDC',
            VehicleTypeCarOptions::PARKING_ASSISTANCE_SYSTEM => 'PDC',
            VehicleTypeCarOptions::AUTOMATIC_AIRCO => 'Climate Control',
        ];
        foreach ($vehicle->get(Vehicle::OPTIONS_ACCESORIES, []) as $option) {
            if (array_key_exists($option, $optionsMapping)) {
                $specials[] = $optionsMapping[$option];
            }
        }

        // Also add the equipments to $specials
        $specials = array_merge($specials, $equipments);

        return array_unique($specials);
    }

    private function getEquipments(Vehicle $vehicle)
    {
        $equipment = [];
        $optionsMapping = [
            VehicleTypeCarOptions::AIRCO => 'airco',
            VehicleTypeCarOptions::ABS => 'abs',
            VehicleTypeCarOptions::FOUR_WHEEL_DRIVE => '4x4',
            VehicleTypeCarOptions::AUTOMATIC_AIRCO => 'automatic_air_conditioning',
            VehicleTypeCarOptions::AIRBAG => 'airbag',
            VehicleTypeCarOptions::TUNING => 'tuning',
            VehicleTypeCarOptions::LEATHER_SEATS => 'leather',
            VehicleTypeCarOptions::AIRBAG_PASSENGER => 'passenger_airbag',
            VehicleTypeCarOptions::METALLIC_RIMS => 'alloy_wheels',
            VehicleTypeCarOptions::ELECTRIC_WINDOWS => 'electric_windows',
            VehicleTypeCarOptions::SIDE_AIRBAGS => 'side_airbags',
            VehicleTypeCarOptions::ROOF_RAILS => 'roofrails',
            VehicleTypeCarOptions::NAVIGATION_SYSTEM => 'navigation',
            VehicleTypeCarOptions::XENON_LIGHTS => 'xenon',
            VehicleTypeCarOptions::ON_BOARD_COMPUTER => 'on_board_computer',
            VehicleTypeCarOptions::ELECTRIC_ADJUSTABLE_SEATS => 'electric_adjustable_seats',
            VehicleTypeCarOptions::CENTRAL_LOCKING => 'central_door_locking',
            VehicleTypeCarOptions::PARKING_ASSISTANCE_SYSTEM => 'park_assist',
            VehicleTypeCarOptions::OPEN_ROOF => 'open_roof',
            VehicleTypeCarOptions::ALARM => 'alarm',
            VehicleTypeCarOptions::FOG_LIGHTS => 'fog_lights',
            VehicleTypeCarOptions::SEAT_HEATING => 'heated_seats',
            VehicleTypeCarOptions::IMMOBILIZER => 'immobilizer',
            VehicleTypeCarOptions::POWER_STEERING => 'power_stearing',
            VehicleTypeCarOptions::CD_PLAYER => 'cd_player',
            VehicleTypeCarOptions::TRACTION_CONTROL => 'traction_control',
            VehicleTypeCarOptions::CRUISE_CONTROL => 'cruise_control',
            VehicleTypeCarOptions::RADIO => 'radio',
            VehicleTypeCarOptions::ESP => 'esp',
            VehicleTypeCarOptions::TOWING_HOOK => 'towbar',
            VehicleTypeCarOptions::PANORAMIC_ROOF => 'panoramaroof',
            VehicleTypeCarOptions::AIR_SUSPENSION => 'air_suspension',
            VehicleTypeCarOptions::START_STOP_SYSTEM => 'start_stop_system',
            VehicleTypeCarOptions::HEADLAMP_SPRAYERS => 'headlight_sprayer',
            VehicleTypeCarOptions::SPORT_CHASSIS => 'sport_suspension',
            VehicleTypeCarOptions::ARMREST_CENTER => 'armrest',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_17_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_18_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_19_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_20_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::ALLOY_WHEEL_RIM_21_INCH => 'alloy_wheels',
            VehicleTypeCarOptions::MULTIFUNCTION_STEERING_WHEEL => 'multi_function_steering_wheel',
            VehicleTypeCarOptions::LED_HEADLIGHTS => 'led_headlights',
            VehicleTypeCarOptions::PRIVACY_GLAS => 'privacy_glass',
            VehicleTypeCarOptions::REAR_CAMERA => 'rear_camera',
            VehicleTypeCarOptions::AUTOMATIC_PARKING => 'automatic_parking',
            VehicleTypeCarOptions::LEVEL_GUIDANCE => 'level_guidance',
            VehicleTypeCarOptions::ADAPTIVE_CRUISE_CONTROL => 'adaptive_cruise_control',
            VehicleTypeCarOptions::ELECTRICALLY_ADJUSTABLE_MIRRORS => 'electrical_adjustable_mirrors',
            VehicleTypeCarOptions::LEATHER_STEERING_WHEEL => 'leather_steering_wheel',
            VehicleTypeCarOptions::LEATHER_KNOB => 'leather_shift_knob',
            VehicleTypeCarOptions::TIRE_PRESSURE_SENSORS => 'tire_pressure_system',
            VehicleTypeCarOptions::KEYLESS_ENTRY => 'keyless_entry',
            VehicleTypeCarOptions::SPORT_CHAIRS => 'sport_seats',
            VehicleTypeCarOptions::FLASHING_LIGHTS_IN_MIRRORS => 'mirror_indicators',
            VehicleTypeCarOptions::CHANGED_RADIO => 'radio',
            VehicleTypeCarOptions::SIDE_BARS => 'side_bars',
            VehicleTypeCarOptions::BULL_BAR => 'bull_bar',
            VehicleTypeCarOptions::HEAD_UP_DISPLAY => 'head_up_display',
            VehicleTypeCarOptions::VIRTUAL_COCKPIT => 'virtual_cockpit',
            VehicleTypeCarOptions::LANE_ASSIST => 'lane_assistent',
            VehicleTypeCarOptions::AUTO_PILOT => 'auto_pilot',
            VehicleTypeCarOptions::CAMERA_360_DEGREES => '360_degrees_camera',
            VehicleTypeCarOptions::ELECTRIC_TAILGATE => 'electric_tailgate',
            VehicleTypeCarOptions::ELECTRIC_TOW_BAR => 'electric_tow_bar',
            VehicleTypeCarOptions::DETACHABLE_TOW_BAR => 'detachable_tow_bar',
            VehicleTypeCarOptions::MASSAGE_SEATS => 'massage_seats',
            VehicleTypeCarOptions::SEAT_VENTILATION => 'seat_ventilation',
            VehicleTypeCarOptions::PARKING_HEATER => 'parking_heater',
            VehicleTypeCarOptions::MEMORY_SEATS => 'memory_seats',
            VehicleTypeCarOptions::PREMIUM_SOUND_SYSTEM => 'premium_sound_system',
        ];

        foreach ($vehicle->get(Vehicle::OPTIONS_ACCESORIES, []) as $option) {
            if (array_key_exists($option, $optionsMapping)) {
                $equipment[] = $optionsMapping[$option];
            }
        }

        $leatherTypes = [Upholstery::LEATHER, Upholstery::SEMI_LEATHER, Upholstery::ALCANTARA];
        if (in_array($vehicle->get(Vehicle::UPHOLSTERY, 0), $leatherTypes, true)) {
            $equipment[] = 'equipment_leather';
        }

        if ($vehicle->get(Vehicle::NUMBER_OF_DOORS) >= 4) {
            $equipment[] = 'four_doors';
        }


        return array_unique($equipment);
    }
}
