<?php

namespace Dealerdirect\Pricing\Providers;

use DateTime;
use DealerDirect\Generic\Category\Fuel;
use Dealerdirect\Pricing\Vehicle;
use Dealerdirect\Generic\Category\VehicleType;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class BidHistoryProvider extends Provider
{
    public function getRequirements(): array
    {
        return [
            Vehicle::BRAND_ID => ['required', 'integer'],
            Vehicle::MODEL_ID => ['required', 'integer'],
            Vehicle::CONSTRUCTION_YEAR => ['required', 'integer'],
            Vehicle::FUEL => ['required', 'integer', Rule::in(Fuel::getAll())],
            Vehicle::ENGINE_CAPACITY => ['integer', 'required_unless:vehicleType,motorbike,scooter'],
            Vehicle::POWER => ['required', 'integer'],
            Vehicle::TYPE => ['required', 'string', Rule::in(VehicleType::getAll())]
        ];
    }

    public function __invoke(Vehicle $vehicle): array
    {
        $this->vehicle = $vehicle;
        $sensibleBids = $this->getSensibleBidsByCollection($this->fetch(), 'bid');

        $data = [
            'average' => (int) $sensibleBids->average('bid'),
            'min' => (int) $sensibleBids->min('bid'), // Lowest bid found in bidhistory
            'max' => (int) $sensibleBids->max('bid'), // Highest bid found in bidhistory
            'num' => (int) $sensibleBids->count(), // Number of total bids found
            // Perhaps add some data concerning age of bids
        ];

        return $data;
    }

    public function getName(): string
    {
        return 'DealerdirectBidHistory';
    }

    private function fetch(): Collection
    {
        $vehicle = $this->vehicle;
        $currentYear = (int) (new DateTime())->format('Y');
        $vehicleType = $vehicle->get(Vehicle::TYPE);
        $isMotor = in_array($vehicleType, [VehicleType::MOTOR, VehicleType::SCOOTER], true);
        $query = $isMotor ? self::MOTOR_QUERY : self::CAR_QUERY;

        $params = [
            'brandId' => $vehicle->get(Vehicle::BRAND_ID),
            'modelId' => $vehicle->get(Vehicle::MODEL_ID),
            'fuelType' => $vehicle->get(Vehicle::FUEL),
            'ageFrom' => $currentYear - $vehicle->get(Vehicle::CONSTRUCTION_YEAR),
            'ageTill' => $currentYear - $vehicle->get(Vehicle::CONSTRUCTION_YEAR),
            'power' => $vehicle->get(Vehicle::POWER),
        ];
        if (!$isMotor) {
            $params['engineCapacity'] = $vehicle->get(Vehicle::ENGINE_CAPACITY);
        }

        $results = DB::connection(config('pricing.bidHistory.connection'))->select($query, $params);

        return collect($results);
    }

    public function isEnabled(): bool
    {
        return true;
    }

    public function isReadOnly(): bool
    {
        return true;
    }

    // TODO: Make this query also available for motorbikes
    private const CAR_QUERY = <<<SQL
        SELECT
            last_bid_biedingen_base.bod AS bid,
            last_bid_biedingen_base.hbieder AS dealer_id,
            last_bid_biedingen_base.datum AS bid_date,
            vb.id AS lot_id
        FROM
            dealer01_ddmain.verzamel_base AS vb
            INNER JOIN (
                SELECT
                    t1.*
                FROM
                    dealer01_ddmain.biedingen_base AS t1
                    LEFT JOIN
                        dealer01_ddmain.biedingen_base AS t2
                        ON t1.kavel = t2.kavel
                        AND t1.status = t2.status
                        AND (
                            t1.datum < t2.datum
                            OR (
                                t1.datum = t2.datum
                                AND t1.tijd < t2.tijd
                            )
                        )
                WHERE
                    t2.kavel IS NULL
                    AND t1.status = 'active'
                    AND YEAR(t1.datum) > (YEAR(NOW()) - 1)
                ) AS last_bid_biedingen_base
                ON last_bid_biedingen_base.kavel = vb.id
            LEFT JOIN
                dealer01_ddmain.bellijst_sort AS bl
                ON bl.kavelref = vb.id
        WHERE
            vb.automerk = :brandId
            AND vb.autotype = :modelId
            AND vb.brandstof = :fuelType
            AND vb.motorinhoud = :engineCapacity
            AND vb.vermogen = :power
            -- TODO: Do something with reference. As in vehicleCountry
            -- AND vb.reference = 1
            AND vb.bouwjaar
                BETWEEN (YEAR(NOW()) - ((:ageFrom + 1) + YEAR(NOW()) - YEAR(vb.datumtijd)))
                AND (YEAR(NOW()) - ((:ageTill - 1) + YEAR(NOW()) - YEAR(vb.datumtijd)))
            AND bl.status != 21
        ORDER BY last_bid_biedingen_base.bod DESC
        LIMIT 50
        SQL;

    private const MOTOR_QUERY = <<<SQL
        SELECT
            last_bid_biedingen_base.bedrag AS bid,
            last_bid_biedingen_base.bedrijvenID AS dealer_id,
            last_bid_biedingen_base.datetime AS bid_date,
            vb.id AS lot_id
        FROM
            dealer01_ddmoto.BO_Motorfiets AS vb
            INNER JOIN (
                SELECT
                    t1.*
                FROM
                    dealer01_ddmoto.BO_Biedhistorie AS t1
                    LEFT JOIN
                        dealer01_ddmoto.BO_Biedhistorie AS t2
                        ON t1.voertuigID = t2.voertuigID
                        AND t1.status = t2.status
                        AND t1.datetime < t2.datetime
                WHERE
                    t2.voertuigID IS NULL
                    AND t1.status = 'active'
                    AND YEAR(t1.datetime) > (YEAR(NOW()) - 1)
                ) AS last_bid_biedingen_base
                ON last_bid_biedingen_base.voertuigID = vb.id
            LEFT JOIN
                dealer01_ddmoto.BO_Bellijst_Sort AS bl
                ON bl.kavelref = vb.id
        WHERE
            vb.merk = :brandId
            AND vb.model = :modelId
            AND vb.brandstof = :fuelType
            -- AND vb.motorinhoud = :engineCapacity
            AND vb.vermogen = :power
            -- TODO: Do something with reference. As in vehicleCountry
            -- AND vb.reference = 1
            AND YEAR(vb.bouwjaar)
                BETWEEN (YEAR(NOW()) - ((:ageFrom + 1) + YEAR(NOW()) - YEAR(vb.aangemaakt)))
                AND (YEAR(NOW()) - ((:ageTill - 1) + YEAR(NOW()) - YEAR(vb.aangemaakt)))
            AND bl.status != 21
        ORDER BY last_bid_biedingen_base.bedrag DESC
        LIMIT 50
        SQL;
}
