<?php

namespace Dealerdirect\Pricing\Providers;

use DealerDirect\Generic\Category\Fuel;
use Dealerdirect\Generic\Category\VehicleType;
use Dealerdirect\Pricing\Providers\Provider;
use Dealerdirect\Pricing\Vehicle;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\Rule;

class DataWorks4YouProvider extends Provider
{
    private const BASE_URL = 'http://api.dataworks4you.nl/api/1';

    public function __invoke(Vehicle $vehicle): array
    {
        $this->vehicle = $vehicle;
        $url = $this->buildUrl('request');
        $response = Http::withHeaders(['API-Key' => config('pricing.dataWorks4You.api_key')])->get($url);
        $xml = simplexml_load_string($response->body());
        $pricing = collect();

        foreach ($xml->children() as $child) {
            $childData = get_object_vars($child);
            if (!empty($childData) && array_key_exists('price', $childData)) {
                $pricing->add((int) $childData['price']);
            }
        }

        $sensibleBids = $this->getSensibleBidsByCollection($pricing);

        return [
            'average' => (int) $sensibleBids->average(),
            'min' => (int) $sensibleBids->min(), // Lowest bid found in bidhistory
            'max' => (int) $sensibleBids->max(), // Highest bid found in bidhistory
            'num' => (int) $sensibleBids->count(), // Number of total bids found
            // Perhaps add some data concerning age of bids
        ];
    }

    public function getName(): string
    {
        return 'DataWorks4You';
    }

    public function isEnabled(): bool
    {
        return true;
    }

    public function isReadOnly(): bool
    {
        return true;
    }

    private function mapper(Vehicle $vehicle): array
    {
        $params = [
            'reference' => 'testing', // Needed for DataWorks4You
            'model' => $this->mapModelName($vehicle),
            'first_reg' => $vehicle->get(Vehicle::CONSTRUCTION_YEAR),
        ];

        if (!in_array($vehicle->get(Vehicle::TYPE), [VehicleType::MOTOR, VehicleType::SCOOTER])) {
            $params['brand'] = $this->mapBrandName($vehicle->get(Vehicle::BRAND));
            $params['engine_power'] = $vehicle->get(Vehicle::POWER);
            $params['fuel'] = $this->mapFuelId($vehicle->get(Vehicle::FUEL));
        }

        return $params;
    }

    private function mapFuelId(int $fuelId)
    {
        $mapping = [
            // Fuel::UNKNOWN => 0,
            Fuel::GASOLINE => 'B',
            Fuel::DIESEL => 'D',
            Fuel::ELECTRICITY => 'E',
            Fuel::LPG => 'G',
            Fuel::LPG_G3 => 'G',
            // Fuel::CRYOGENIC => 6,
            // Fuel::CNG => 7,
            // Fuel::HYDROGEN => 8,
            Fuel::HYBRID_GASOLINE => 'PHB',
            Fuel::HYBRID_DIESEL => 'PHD',
            Fuel::HYBRID_GASOLINE_LPG => 'PHB',

            // RESPONSE (source):
            // "Aardgas" => 898203
            // "Benzine" => 898200
            // "Diesel" => 898205
            // "Elektrisch" => 898206
            // "Ethanol" => 1640030
            // "LPG" => 898207
            // "Hybride" => 898209
            // "Overig" => 898848
            // "Plug-in Hybride Benzine" => 898215
            // "Plug-in Hybride Diesel" => 898216
            // "Super" => 898211
            // "Waterstof" => 898212
        ];

        return $mapping[$fuelId] ?? null;
    }

    private function mapBrandName(string $brandName)
    {
        return $brandName;
    }

    private function mapModelName(Vehicle $vehicle)
    {
        $modelName = $vehicle->get(Vehicle::MODEL);
        $modelMapping = [
            '1-SERIE' => '1 Serie',
            '2-SERIE' => '2 Serie',
            '3-SERIE' => '3 Serie',
            '4-SERIE' => '4 Serie',
            '5-SERIE' => '5 Serie',
            '6-SERIE' => '6 Serie',
            '7-SERIE' => '7 Serie',
            '8-SERIE' => '8 Serie',
            'SUPERDUKE 1290 R' => '1290 SUPER DUKE R',
        ];

        return $modelMapping[$modelName] ?? $modelName;
    }

    public function getRequirements(): array
    {
        return [
            Vehicle::TYPE => ['required', 'string', Rule::in(VehicleType::getAll())],
            Vehicle::BRAND => ['required', 'string'],
            Vehicle::MODEL => ['required', 'string'],
            Vehicle::CONSTRUCTION_YEAR => ['required', 'integer'],
            Vehicle::POWER => ['required', 'integer'],
            Vehicle::FUEL => ['required', 'integer'],
        ];
    }

    private function buildUrl(string $url)
    {
        $params = [];
        foreach ($this->mapper($this->vehicle) as $key => $value) {
            if (!empty($value)) {
                $params[] = "$key=$value";
            }
        }

        return sprintf('%s/%s?%s', self::BASE_URL, trim($url, '/'), implode('&', $params));
    }
}
