<?php

namespace DealerDirect\Sdk\Facade;

use Dealerdirect\EmailServiceSdk\Api;
use Dealerdirect\EmailServiceSdk\ApiException;
use Dealerdirect\EmailServiceSdk\Model;
use DealerDirect\Sdk\Exception\HttpException;
use Dealerdirect\Sdk\Model\Email\Attachment;
use Dealerdirect\Sdk\Model\Email\Email as EmailModel;
use Dealerdirect\Sdk\Model\Email\Recipient;

/**
 * Facade for the Email SDK
 *
 * @TODO: Eventually the conversions should be moved to a separate class and injected into the facade
 */
class Email
{
    const ERROR_MODEL_UNKNOWN = 'No conversion available for model of type %s';
    const ERROR_EMAIL_FAILURE = 'Could not send email';

    /** @var Api\SendEmailApi */
    private $sendEmailApi;

    final public function __construct(Api\SendEmailApi $sendEmailApi)
    {
        $this->sendEmailApi = $sendEmailApi;
    }

    /**
     * @param $email
     * @param Recipient[] $recipients
     *
     * @throws \DealerDirect\Sdk\Exception\HttpException
     * @throws \UnexpectedValueException
     *
     * @FIXME: With the current implementation BCC and CC are not supported! 2017/09/14/BMP
     */
    final public function sendEmail(EmailModel $email, array $recipients)
    {
        $payload = new Model\EmailPayload([
            'email' => new Model\Email([
                'locale' => $email->getLocale(),
                'emailType' => $email->getEmailType(),
                'lotType' => $email->getVehicleType(),
                'substitutions' => $email->getSubstitution(),
                'to' => $this->convertModels($recipients),
                'attachments' => $this->convertModels($email->getAttachments()),
            ])
        ]);

        try {
            $this->sendEmailApi->emailsPost($payload);
        } catch (ApiException $apiException) {
            throw $this->convertException($apiException);
        }
    }

    /**
     * @param ApiException $exception
     *
     * @return HttpException
     */
    private function convertException(ApiException $exception)
    {
        $messages = $this->retrieveErrorsFromResponse($exception->getResponseBody());

        $glue = PHP_EOL . '- ';

        $message = sprintf(
            '%s. Encountered %d error(s): %s',
            self::ERROR_EMAIL_FAILURE,
            count($messages),
            $glue . implode($glue, $messages)
        );

        return new HttpException($message, $exception->getCode(), $exception);
    }

    /**
     * @param array $subjects
     *
     * @return array
     *
     * @throws \UnexpectedValueException
     */
    private function convertModels(array $subjects)
    {
        $models = [];

        if (count($subjects) > 0) {
            $subject = reset($subjects);

            if (is_object($subject) === true) {
                $type = get_class($subject);

                switch ($type) {
                    case 'Dealerdirect\\Sdk\\Model\\Email\\Attachment':
                        $models = $this->convertAttachmentsToModel($subjects);
                        break;

                    case 'Dealerdirect\\Sdk\\Model\\Email\\Recipient':
                        $models = $this->convertRecipientsToModel($subjects);
                        break;

                    default:
                        throw new \UnexpectedValueException(sprintf(self::ERROR_MODEL_UNKNOWN, $type));
                }
            } else {
                $type = gettype($subject);
                throw new \UnexpectedValueException(sprintf(self::ERROR_MODEL_UNKNOWN, $type));

            }
        }

        return $models;
    }

    /**
     * @param Attachment[] $emailAttachments
     *
     * @return Model\Attachment[]
     */
    private function convertAttachmentsToModel(array $emailAttachments)
    {
        $attachments = [];

        array_walk($emailAttachments, function (Attachment $attachment) use (&$attachments) {
            $attachments[] = new Model\Attachment([
                'content' => base64_encode($attachment->getContent()),
                'disposition' => $attachment->getDisposition(),
                'fileName' => $attachment->getFileName(),
                'mimeType' => $attachment->getMimeType(),
            ]);
        });

        return $attachments;
    }

    /**
     * @param Recipient[] $recipients
     *
     * @return Model\To[]
     */
    private function convertRecipientsToModel(array $recipients)
    {
        $to = [];

        array_walk($recipients, function (Recipient $recipient) use (&$to) {
            $to[] = new Model\To([
                'address' => $recipient->getEmailAddress(),
                'name' => $recipient->getName(),
            ]);
        });

        return $to;
    }

    /**
     * @param array $haystack
     *
     * @return array
     */
    private function removeEmptyValuesFromArray(array $haystack)
    {
        foreach ($haystack as $key => $value) {
            if (is_array($value)) {
                $haystack[$key] = $this->removeEmptyValuesFromArray($haystack[$key]);
            }

            if (empty($haystack[$key])) {
                unset($haystack[$key]);
            }
        }

        return $haystack;
    }

    /**
     * @param \stdClass $response
     *
     * @return array
     */
    private function retrieveErrorsFromResponse(\stdClass $response)
    {
        $errors = [];

        if (
            property_exists($response, 'error')
            && property_exists($response->error, 'children')
        ) {
            /* @NOTE: Convert object recursively to array */
            $errorMessages = json_decode(json_encode($response->error), true);

            $errorMessages = $this->removeEmptyValuesFromArray($errorMessages['children']);

            $query = http_build_query($errorMessages);
            $messages = explode('&', $query);
            array_walk($messages, function ($value) use (&$errors) {
                $decodedQuery = urldecode($value);
                $value = preg_replace('/\[(children|errors|[0-9]+)\]/', '', $decodedQuery);
                $errors[] = ucfirst($value);
            });
        }

        return $errors;
    }
}
