<?php

namespace DealerDirect\Sdk\Facade;

use DealerDirect\BrokerageAgreement\Entity\BrokerageAgreement as BrokerageAgreementEntity;
use DealerDirect\Sdk\Client\HttpClient;
use DealerDirect\Sdk\Exception\ServerException;
use GuzzleHttp\Exception\ParseException;
use GuzzleHttp\Message\ResponseInterface;

class BrokerageAgreement
{
    const ERROR_RESPONSE_EMPTY = 'Received an empty response from the REST API';
    const ERROR_RESPONSE_NOT_JSON = 'Response is not valid JSON: "%s"';
    const ERROR_RESPONSE_NOT_JSON_ARRAY = 'Response is not a JSON array: "%s"';

    const URL_PREFIX = 'brokerage-agreement-api';

    /** @var string */
    private $errorMessage = '';
    /** @var HttpClient */
    private $httpClient;
    /** @var bool */
    private $throwExceptions = true;

    /** @return string */
    public function getErrorMessage()
    {
        return $this->errorMessage;
    }

    /** @param string $errorMessage */
    public function setErrorMessage($errorMessage)
    {
        $this->errorMessage = $errorMessage;
    }

    final public function __construct(HttpClient $httpClient)
    {
        $this->httpClient = $httpClient;
    }

    /**
     *
     * @param BrokerageAgreementEntity $entity
     *
     * @return BrokerageAgreementEntity
     *
     * @throws \DealerDirect\Sdk\Exception\ServerException
     * @throws \LogicException
     * @throws \RuntimeException
     * @throws \GuzzleHttp\Exception\RequestException
     */
    final public function create(BrokerageAgreementEntity $entity)
    {
        $httpClient = $this->httpClient;

        return $this->call(
            $httpClient::POST,
            '/create',
            ['json' => $entity->toArray()]
        );
    }

    /**
     *
     * @param string $id
     *
     * @return BrokerageAgreementEntity
     * @throws \DealerDirect\Sdk\Exception\ServerException
     *
     * @throws \LogicException
     * @throws \RuntimeException
     * @throws \GuzzleHttp\Exception\RequestException
     */
    final public function read($id)
    {
        $httpClient = $this->httpClient;

        return $this->call(
            $httpClient::GET,
            sprintf('/read/%s', $id)
        );
    }

    /**
     *
     * @param BrokerageAgreementEntity $entity
     *
     * @return BrokerageAgreementEntity
     * @throws \DealerDirect\Sdk\Exception\ServerException
     *
     * @throws \LogicException
     * @throws \RuntimeException
     * @throws \GuzzleHttp\Exception\RequestException
     */
    final public function update(BrokerageAgreementEntity $entity)
    {
        $httpClient = $this->httpClient;

        return $this->call(
            $httpClient::POST,
            '/update',
            ['json' => $entity->toArray()]
        );
    }

    /**
     * @param string $id
     *
     * @return BrokerageAgreementEntity
     * @throws \DealerDirect\Sdk\Exception\ServerException
     *
     * @throws \LogicException
     * @throws \RuntimeException
     * @throws \GuzzleHttp\Exception\RequestException
     */
    final public function sendReminderMail($id)
    {
        $httpClient = $this->httpClient;

        return $this->call(
            $httpClient::GET,
            sprintf('/send-reminder-mail/%s', $id)
        );
    }

    /**
     * @param string $method
     * @param string $uri
     * @param array $options
     *
     * @return BrokerageAgreementEntity|bool
     * @throws \DealerDirect\Sdk\Exception\ServerException
     *
     * @throws \LogicException
     * @throws \RuntimeException
     * @throws \GuzzleHttp\Exception\RequestException
     */
    private function call($method, $uri, array $options = [])
    {
        $result = null;

        $url = self::URL_PREFIX .$uri;

        $response = $this->httpClient->request($method, $url, $options);

        $statusCode = $response->getStatusCode();

        if ($statusCode < 200 || $statusCode > 299) {
            $this->handleErrorResponse($response);
        } else {

            try {
                $jsonResponse = $response->json();
            } catch (ParseException $exception) {
                $contents = $response->getBody()->getContents();
                throw new ServerException(sprintf(self::ERROR_RESPONSE_NOT_JSON, $contents));
            }

            if (is_array($jsonResponse) === false) {
                $contents = $response->getBody()->getContents();

                if ($contents === '') {
                    throw new ServerException(self::ERROR_RESPONSE_EMPTY);
                } else {
                    throw new ServerException(sprintf(self::ERROR_RESPONSE_NOT_JSON_ARRAY, $contents));
                }
            }

            /** @var array $jsonResponse */
            if (array_key_exists('data', $jsonResponse)
                && is_array($jsonResponse['data'])
                && array_key_exists('brokerage-agreement', $jsonResponse['data'])
            ) {
                // @CHECKME: Only _one_ element should be returned, should this be validated?
                $entityArray = array_shift($jsonResponse['data']);
                $result = BrokerageAgreementEntity::fromArray($entityArray);
            } else {
                throw new ServerException(sprintf('Could not find entity in JSON response: "%s"', json_encode($jsonResponse, JSON_FORCE_OBJECT)));
            }
        }

        return $result;
    }

    /**
     * @param ResponseInterface $response
     *
     * @throws \RuntimeException
     */
    private function handleErrorResponse(ResponseInterface $response)
    {
        $statusCode = $response->getStatusCode();
        $jsonResponse = $response->json();

        $message = self::ERROR_RESPONSE_EMPTY;

        if (is_array($jsonResponse)
            && array_key_exists('error', $jsonResponse)
        ) {
            if (is_array($jsonResponse['error'])
                && array_key_exists('message', $jsonResponse['error'])
            ) {
                $message = $jsonResponse['error']['message'];
            } else {
                $message = var_export($jsonResponse['error'], true);
            }
        } elseif ($response->getBody()->getContents() !== '') {
            $message = $response->getBody()->getContents();
        }/* ♪♬ and nothing else matters ♫♩ */

        if ($this->throwExceptions === true) {
            if ($statusCode >= 400 && $statusCode <= 499) {
                $exception = '\\DealerDirect\\Sdk\\Exception\\ClientException';
            } elseif ($statusCode >= 500 && $statusCode <= 599) {
                $exception = '\\DealerDirect\\Sdk\\Exception\\ServerException';
            } else {
                $exception = '\\DealerDirect\\Sdk\\Exception\\HttpException';
            }
            throw new $exception($message);
        } else {
            $this->setErrorMessage($message);
        }
    }
}

/*EOF*/
