<?php

namespace Dealerdirect\SMS;

use Dealerdirect\SMS\Exceptions\SmsException;
use Dealerdirect\SMS\Messages\Message;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberType;
use libphonenumber\PhoneNumberUtil;
use Vonage\Client;
use Vonage\SMS\Message\SMS as SmsMessage;

class SMS
{
    /** @var Client */
    private $client;
    /** @var Message */
    private $message;

    /**
     * SMS Constructor
     *
     * @param string $apiKey
     * @param string $apiSecret
     * @param Message $message
     */
    public function __construct(Client $client, Message $message)
    {
        $this->client = $client;
        $this->message = $message;
    }

    /**
     * @param array $phoneNumbers
     * @return string The phone number where the sms has been sent to.
     * @throws \Exception
     */
    public function send(array $phoneNumbers): string
    {
        $mobile = $this->findMobileAndFormat($phoneNumbers);

        try {
            $sms = $this->client->sms()->send(
                new SmsMessage($mobile, $this->message->getSender(), $this->message->getContent())
            );
        } catch (\Exception $exception) {
            throw SmsException::request($exception);
        }

        if (!$sms->valid()) {
            throw SmsException::invalidResponse();
        }

        $status = $sms->current()->getStatus();
        if ($status !== 0) {
            throw SmsException::invalidStatus($status);
        }

        return $mobile;
    }

    /**
     * Gets the first mobile phone number from array
     * Throws exception when no mobile phone was found.
     *
     * @param array $phoneNumbers
     * @return string The first mobile phone number found in $phoneNumbers
     * @throws \Exception
     */
    private function findMobileAndFormat(array $phoneNumbers, bool $withCountry = true): string
    {
        $phoneUtil = PhoneNumberUtil::getInstance();
        $country = $withCountry ? $this->message->getCountry() : null;
        foreach ($phoneNumbers as $number) {
            try {
                $parsedPhone = $phoneUtil->parse($number, $country);
                if ($phoneUtil->getNumberType($parsedPhone) === PhoneNumberType::MOBILE) {
                    return $phoneUtil->format($parsedPhone, PhoneNumberFormat::E164);
                }
            } catch (NumberParseException $exception) {
                // Fail silence
            }
        }

        if ($withCountry) {
            // Retry this to find a mobile number without country
            return $this->findMobileAndFormat($phoneNumbers, false);
        }

        throw SmsException::noMobileNumber($phoneNumbers);
    }
}
