<?php

namespace Dealerdirect\SMS;

use Dealerdirect\SMS\Exceptions\MessageException;
use Vonage\Client;
use Vonage\Client\Credentials\Basic;

class Helper
{
    /**
     * Gets SMS client.
     *
     * @return Client
     */
    public static function getClient(string $apiKey, string $apiSecret): Client
    {
        return new Client(
            new Basic($apiKey, $apiSecret)
        );
    }

    /**
     * Pulls value from array with given key.
     *
     * @param array $data
     * @param mixed $key
     * @return void
     */
    public static function pullFromArray(array &$data, $key)
    {
        $value = $data[$key] ?? null;
        unset($data[$key]);

        return $value;
    }

    /**
     * Validates data with required keys. Only checks if value not is empty.
     *
     * @param array $requiredKeys
     * @param array $data
     * @return void
     * @throws \Exception
     */
    public static function validateData(array $requiredKeys, array $data): void
    {
        $errors = array_filter($requiredKeys, fn($key) => empty($data[$key]), ARRAY_FILTER_USE_BOTH);
        if (!empty($errors)) {
            MessageException::missingAttributes($errors);
        }
    }

    /**
     * Validates if branch is available in data.
     *
     * @param string $branch
     * @param array $data
     * @return void
     * @throws \Exception
     */
    public static function validateBranch(string $branch, array $data)
    {
        if (!array_key_exists($branch, $data)) {
            throw MessageException::invalidBranch($branch, $data);
        }
    }

    /**
     * Validates if branch is available in data.
     *
     * @param string $branch
     * @param array $data
     * @return void
     * @throws \Exception
     */
    public static function validateCountry(string $country, array $data)
    {
        if (!array_key_exists($country, $data)) {
            throw MessageException::invalidCountry($country, $data);
        }
    }
}
