<?php

namespace Dealerdirect\SMS\Messages;

use Dealerdirect\SMS\Helper;
use DealerDirect\Generic\Category\Territory;
use Dealerdirect\SMS\Messages\Traits\Getters;

class AgreeSale implements Message
{
    use Getters;

    /** @var string */
    private $branch;
    /** @var array */
    private $data;

    private const SENDER_DATA = [
        'car_nl' => [
            'sender' => 'IWVMAA',
            'website' => 'www.ikwilvanmijnautoaf.nl',
            'phone' => '053-7600027',
        ],
        'car_be' => [
            'sender' => 'IWVMAA',
            'website' => 'www.ikwilvanmijnautoaf.be',
            'phone' => '02 / 80.80.146',
        ],
        'motorbike_nl' => [
            'sender' => 'IWVMMA',
            'website' => 'www.ikwilvanmijnmotorfietsaf.nl',
            'phone' => '053 - 208 4449',
        ],
        'motorbike_be' => [
            'sender' => 'IWVMMA',
            'website' => 'www.ikwilvanmijnmotoraf.be',
            'phone' => '016-980331',
        ],
        'motorbike_de' => [
            'sender' => 'IWMMLW',
            'website' => 'www.ichwillmeinmotorradloswerden.de',
            'phone' => '02131-7514670',
        ],
        'scooter_nl' => [
            'sender' => 'IWVMSA',
            'website' => 'www.ikwilvanmijnscooteraf.nl',
            'phone' => '053 - 208 4449',
        ],
        'bicycle_nl' => [
            'sender' => 'IWVMFA',
            'website' => 'www.ikwilvanmijnfietsaf.nl',
            'phone' => '053 - 208 4444',
        ],
        'marktplaats_nl' => [
            'sender' => '+31857733443',
            'phone' => '+31857733443',
        ],
    ];

    /**
     * @inheritDoc
     */
    public function __construct(string $vehicleType, string $country, array $data)
    {
        $this->vehicleType = $vehicleType;
        $this->country = $country;
        $this->branch = strtolower("{$this->vehicleType}_{$this->country}");

        Helper::validateData(['make', 'model', 'emailAddress'], $data);
        Helper::validateBranch($this->branch, self::SENDER_DATA);

        $senderData = self::SENDER_DATA[$this->branch];
        $this->sender = Helper::pullFromArray($senderData, 'sender');
        $this->data = $data + $senderData;
    }

    /**
     * @inheritDoc
     */
    public function getContent(): string
    {
        $data = $this->data;
        if ($this->vehicleType === 'marktplaats') {
            return <<<TXT
            Gefeliciteerd met de verkoop van jouw {$data['make']} {$data['model']}.
            Je ontvangt de verkoopovereenkomst op {$data['emailAddress']}.
            Vragen? {$data['phone']}
            TXT;
        }

        $bodies = [
            Territory::NL => <<<TXT
                Gefeliciteerd met de verkoop van jouw {$data['make']} {$data['model']}.
                Je ontvangt de verkoopovereenkomst op {$data['emailAddress']}.
                Vragen? {$data['phone']}, {$data['website']}
            TXT,
            Territory::BE => <<<TXT
                Gefeliciteerd met de verkoop van uw {$data['make']} {$data['model']}.
                U ontvangt de verkoopovereenkomst op {$data['emailAddress']}.
                Vragen? Bel ons op {$data['phone']} of bezoek onze website: {$data['website']}
            TXT,
            Territory::DE => <<<TXT
                Wir gratulieren Ihnen zum Verkauf Ihrer {$data['make']} {$data['model']}.
                Wir senden Ihnen den Kaufvertrag per E-Mail an {$data['emailAddress']}.
                Falls Sie noch Fragen haben, rufen Sie uns unter {$data['phone']} an
                oder besuchen Sie unsere Website: {$data['website']}
            TXT,
        ];

        Helper::validateCountry($this->country, $bodies);

        return $bodies[$this->country];
    }
}
