<?php

namespace Dealerdirect\SMS\Messages;

use Dealerdirect\SMS\Helper;
use DealerDirect\Generic\Category\Territory;
use Dealerdirect\SMS\Messages\Traits\Getters;

class AgreeSale implements Message
{
    use Getters;

    /** @var string */
    private $branch;
    /** @var array */
    private $data;

    private const SENDER_DATA = [
        'car_nl' => [
            'sender' => 'IWVMAA',
            'website' => 'www.ikwilvanmijnautoaf.nl',
        ],
        'car_be' => [
            'sender' => 'IWVMAA',
            'website' => 'www.ikwilvanmijnautoaf.be',
        ],
        'motorbike_nl' => [
            'sender' => 'IWVMMA',
            'website' => 'www.ikwilvanmijnmotorfietsaf.nl',
        ],
        'motorbike_be' => [
            'sender' => 'IWVMMA',
            'website' => 'www.ikwilvanmijnmotoraf.be',
        ],
        'motorbike_de' => [
            'sender' => 'IWMMLW',
            'website' => 'www.ichwillmeinmotorradloswerden.de',
        ],
        'scooter_nl' => [
            'sender' => 'IWVMSA',
            'website' => 'www.ikwilvanmijnscooteraf.nl',
        ],
        'bicycle_nl' => [
            'sender' => 'IWVMFA',
            'website' => 'www.ikwilvanmijnfietsaf.nl',
        ],
        'marktplaats_nl' => [
            'sender' => 'Marktplaats', // Will be overwritten in the constructor with the actual number
        ],
    ];

    /**
     * @inheritDoc
     */
    public function __construct(string $vehicleType, string $country, array $data)
    {
        $this->vehicleType = $vehicleType;
        $this->country = $country;
        $this->branch = strtolower("{$this->vehicleType}_{$this->country}");

        Helper::validateData(['make', 'model', 'emailAddress', 'phone'], $data);
        Helper::validateBranch($this->branch, self::SENDER_DATA);

        $senderData = self::SENDER_DATA[$this->branch];
        if ($this->vehicleType === 'marktplaats') {
            $senderData['sender'] = $data['phone'];
        }

        $this->sender = Helper::pullFromArray($senderData, 'sender');
        $this->data = $data + $senderData;
    }

    /**
     * @inheritDoc
     */
    public function getContent(): string
    {
        $data = $this->data;
        if ($this->vehicleType === 'marktplaats') {
            return <<<TXT
            Gefeliciteerd met de verkoop van jouw {$data['make']} {$data['model']}.
            Je ontvangt de verkoopovereenkomst op {$data['emailAddress']}.
            Vragen? {$data['phone']}
            TXT;
        }

        $bodies = [
            Territory::NL => <<<TXT
                Gefeliciteerd met de verkoop van jouw {$data['make']} {$data['model']}.
                Je ontvangt de verkoopovereenkomst op {$data['emailAddress']}.
                Vragen? {$data['phone']}, {$data['website']}
            TXT,
            Territory::BE => <<<TXT
                Gefeliciteerd met de verkoop van uw {$data['make']} {$data['model']}.
                U ontvangt de verkoopovereenkomst op {$data['emailAddress']}.
                Vragen? Bel ons op {$data['phone']} of bezoek onze website: {$data['website']}
            TXT,
            Territory::DE => <<<TXT
                Wir gratulieren Ihnen zum Verkauf Ihrer {$data['make']} {$data['model']}.
                Wir senden Ihnen den Kaufvertrag per E-Mail an {$data['emailAddress']}.
                Falls Sie noch Fragen haben, rufen Sie uns unter {$data['phone']} an
                oder besuchen Sie unsere Website: {$data['website']}
            TXT,
        ];

        Helper::validateCountry($this->country, $bodies);

        return $bodies[$this->country];
    }
}
