<?php
namespace DealerDirect\VehicleClassification;

use DealerDirect\VehicleClassification\Models\FuelUtils;
use DealerDirect\VehicleClassification\Models\Mappers\VehicleModelMapper;
use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;

/**
 * Class VehicleClassificationClassTest
 * @package DealerDirect\VehicleClassification\Tests
 */
class VehicleClassificationClassTest extends \PHPUnit_Framework_TestCase
{

    /**
     * @param $vehicleData
     *
     * @dataProvider provideDataGroupMotorYoung
     */
    public function testVehicleClassificationAllOnMotorGroupYoung(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $vehicleClassification = VehicleClassification::create(DefaultGroupRulesSetsFactory::getDefaultMotoGroupRulesSet());

        $classifiedId = $vehicleClassification->suggestClassifications($vehicle);

        $this->assertEquals(GroupRulesInterface::GROUP_MOTORS_YOUNG_ID, $classifiedId[0]);
    }

    /**
     * @param $vehicleData
     *
     * @dataProvider provideDataGroupMotorOther
     */
    public function testVehicleClassificationAllOnMotorGroupOther(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $vehicleClassification = VehicleClassification::create(DefaultGroupRulesSetsFactory::getDefaultMotoGroupRulesSet());

        $classifiedId = $vehicleClassification->suggestClassifications($vehicle);

        $this->assertEquals(GroupRulesInterface::GROUP_MOTORS_OTHER_ID, $classifiedId[0]);
    }

    /**
     * @param $vehicleData
     *
     * @dataProvider provideDataGroupCarExport
     */
    public function testVehicleClassificationAllOnCarGroupExport(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $vehicleClassification = VehicleClassification::create(DefaultGroupRulesSetsFactory::getDefaultCarGroupRulesSet());

        $classifiedId = $vehicleClassification->suggestClassifications($vehicle);

        $this->assertEquals(GroupRulesInterface::GROUP_CARS_EXPORT_ID, $classifiedId[0]);
    }

    /**
     * @param $vehicleData
     *
     * @dataProvider provideDataGroupCarPremiumExclusive
     */
    public function testVehicleClassificationAllOnCarGroupPremiumExclusive(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $vehicleClassification = VehicleClassification::create(DefaultGroupRulesSetsFactory::getDefaultCarGroupRulesSet());

        $classifiedId = $vehicleClassification->suggestClassifications($vehicle);

        $this->assertEquals(GroupRulesInterface::GROUP_CARS_PREMIUM_EXCLUSIVE_ID, $classifiedId[0]);
    }

    /**
     * @param array $vehicleData
     *
     * @dataProvider provideDataChecklistLowHangingCarNl
     */
    public function testVehicleClassificationCarNlOnChecklistLowHanging(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        $vehicleClassification = VehicleClassification::create(
            DefaultGroupRulesSetsFactory::getDefaultCarNlGroupRulesSet()
        );

        $classifiedId = $vehicleClassification->suggestClassifications($vehicle);

        $this->assertEquals(GroupRulesInterface::GROUP_NO_ID, $classifiedId[0]);
    }

    public static function provideDataChecklistLowHangingCarNl()
    {
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'mileage' => 204000,
                'makeName' => 'AUDI',
                'modelName' => 'A3',
                'yearOfManufacture' => 2002,
                'fuelType' => '1',
                'bpm' => '409',
                'valuationPrice' => 300,
                'numberOfDoors' => 5,
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'makeName' => 'PEUGEOT',
                'modelName' => '206',
                'fuelType' => 'benzine',
                'mileage' => '15000',
                'yearOfManufacture' => 2000,
                'bpm' => '2729',
                'con_carrosserie' => '3',
                'damageFree' => true,
                'drivable' => true,
                'valuationPrice' => 349,
                'carrosserie' => '1',
                'uitvoering' => '206 Gentry 1.4',
                'motorinhoud' => '1360',
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'makeName' => 'TOYOTA',
                'modelName' => 'Corolla',
                'fuelType' => FuelUtils::DIESEL,
                'mileage' => 204000,
                'bpm' => 409,
                'valuationPrice' => 100,
                'numberOfDoors' => 3,
            )),
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideDataGroupMotorYoung()
    {
        $currentYear = (int)date('Y');
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 14507,
                'makeName' => 'PIAGGIO',
                'modelName' => 'M64',
                'yearOfManufacture' => $currentYear-8,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4750
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 22000,
                'makeName' => 'YAMAHA',
                'modelName' => 'XT 660 R',
                'yearOfManufacture' => $currentYear-6,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 3300
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 19700,
                'makeName' => 'DUCATI',
                'modelName' => 'MONSTER 796',
                'yearOfManufacture' => $currentYear-6,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 6000
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 11000,
                'makeName' => 'BENZHOU',
                'modelName' => 'IVA LUX50',
                'yearOfManufacture' => $currentYear-6,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 650
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 25850,
                'makeName' => 'HONDA',
                'modelName' => 'NC700D',
                'yearOfManufacture' => 2012,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4250
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 38000,
                'makeName' => 'BMW',
                'modelName' => 'R 1150 GS',
                'yearOfManufacture' => $currentYear-16,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 0
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 82136,
                'makeName' => 'HARLEY-DAVIDSON',
                'modelName' => 'FLHT - FLHTI ELECTRA GLIDE STANDARD ',
                'yearOfManufacture' => $currentYear-21,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4400
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 30260,
                'makeName' => 'BMW',
                'modelName' => 'R 850 R',
                'yearOfManufacture' => $currentYear-11,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4000
            )),
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideDataGroupMotorOther()
    {
        $currentYear = (int)date('Y');
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '2',
                'mileage' => 3600,
                'makeName' => 'PIAGGIO',
                'modelName' => 'FLY',
                'yearOfManufacture' => $currentYear-11,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 75
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 14000,
                'makeName' => 'YAMAHA',
                'modelName' => 'VM04',
                'yearOfManufacture' => $currentYear-9,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 6500
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 15000,
                'makeName' => 'PIAGGIO',
                'modelName' => 'M34',
                'yearOfManufacture' => $currentYear-13,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 2800
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => true,
                'mileage' => 79000,
                'makeName' => 'BMW',
                'modelName' => '328 IA',
                'yearOfManufacture' => $currentYear-12,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4000
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => true,
                'mileage' => 75000,
                'makeName' => 'HONDA',
                'modelName' => 'PAN EUROPEAN',
                'yearOfManufacture' => $currentYear-12,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 5000
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => true,
                'mileage' => 75000,
                'makeName' => 'HONDA',
                'modelName' => 'VALKYRIE INTERSTATE',
                'yearOfManufacture' => $currentYear-10,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 10000
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 45000,
                'makeName' => 'YAMAHA',
                'modelName' => 'VM04',
                'yearOfManufacture' => $currentYear-11,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 6500
            )),
            // Classics:
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 89000,
                'makeName' => 'YAMAHA',
                'modelName' => 'Classic1',
                'yearOfManufacture' => 1985,
                'classic' => '1',
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 2500
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => '1',
                'mileage' => 89000,
                'makeName' => 'HONDA',
                'modelName' => 'Classic2',
                'yearOfManufacture' => 1976,
                'classic' => '0',
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 1500
            )),
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideDataGroupMotorDemolition()
    {
        $currentYear = (int)date('Y');
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => false,
                'mileage' => 62000,
                'makeName' => 'PIAGGIO',
                'modelName' => 'FLY',
                'yearOfManufacture' => $currentYear-11,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 75
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => false,
                'mileage' => 82000,
                'makeName' => 'PIAGGIO',
                'modelName' => 'FLY',
                'yearOfManufacture' => $currentYear-11,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 75
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => false,
                'mileage' => 100000,
                'makeName' => 'YAMAHA',
                'modelName' => 'VM04',
                'yearOfManufacture' => $currentYear-12,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 4500
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => false,
                'mileage' => 81000,
                'makeName' => 'PIAGGIO',
                'modelName' => 'M34',
                'yearOfManufacture' => $currentYear-19,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 2800
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_MOTOR,
                'damageFree' => true,
                'mileage' => 62000,
                'makeName' => 'YAMAHA',
                'modelName' => '100',
                'yearOfManufacture' => 1984,
                'fuelType' => '1',
                'bpm' => '0',
                'askingPrice' => 750
            )),
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideDataGroupCarExport()
    {
        return array(
            array(array(
                'id' => '1',
                'licensePlate' => '44NJTB',
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '1',
                'mileage' => 316000,
                'makeName' => 'SAAB',
                'modelName' => '9-3',
                'yearOfManufacture' => 2003,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => '10156',
                'numberOfDoors' => 5,
                'askingPrice' => '1650'
            ))
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideDataGroupCarPremiumExclusive()
    {
        return array(
            array(array(
                'id' => '27',
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'mileage' => 144000,
                'makeName' => 'SPYKER',
                'yearOfManufacture' => 1979,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => 11000,
                'askingPrice' => 15000,
                'actualValue' => 10000,
                'numberOfDoors' => 2
            )),
            array(array(
                'id' => '28',
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'mileage' => 77000,
                'makeName' => 'MORGAN',
                'yearOfManufacture' => 1966,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => 11000,
                'askingPrice' => 39000,
                'actualValue' => 40000,
                'numberOfDoors' => 2
            )),
            array(array(
                'id' => '29',
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'mileage' => 44000,
                'makeName' => 'PORSCHE',
                'modelName' => '911 TURBO',
                'yearOfManufacture' => 1986,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => 0,
                'askingPrice' => 72000,
                'listPrice' => 100000,
                'numberOfDoors' => 2
            )),
            array(array(
                'id' => '30',
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'mileage' => 86000,
                'makeName' => 'PORSCHE',
                'modelName' => 'CARRERA',
                'yearOfManufacture' => 1999,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => 8000,
                'askingPrice' => 65000,
                'listPrice' => 89000,
                'numberOfDoors' => 2
            )),
            array(array(
                'id' => '31',
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'mileage' => 30000,
                'makeName' => 'TESLA',
                'modelName' => 'MODEL S',
                'yearOfManufacture' => 2015,
                'fuelType' => FuelUtils::ELECTRICITY,
                'bpm' => 8000,
                'askingPrice' => 62000,
                'listPrice' => 79000,
                'numberOfDoors' => 4
            )),
        );
    }
}
