<?php
namespace Dealerdirect\VehicleClassification\Rules\Motors;

use Dealerdirect\VehicleClassification\Models\Vehicle;
use Dealerdirect\VehicleClassification\Models\VehicleTypeMotorBodyType as BodyType;
use Dealerdirect\VehicleClassification\Rules\Operators\CatchAll;
use Dealerdirect\VehicleClassification\Rules\RuleBuilder;
use Dealerdirect\VehicleClassification\Rules\RuleSet as BaseRuleSet;
use Ruler\Operator\LogicalAnd;
use Ruler\Operator\LogicalOr;

class RuleSet extends BaseRuleSet
{

    const GROUP_NO_ID = 0;

    // Motors group
    const GROUP_YOUNG_ID = 40;
    const GROUP_OTHER_ID = 41;
    const GROUP_DEMOLITION_ID = 42;
    const GROUP_BUDGET_ID = 43;
    const GROUP_SCOOTER_ID = 44;

    public function __construct()
    {
        $rb = new RuleBuilder();
        $currentYear = (int)date('Y');

        // 1. Alles met een motorinhoud tot 50cc -> Brom & Snor:
        // 2. Voertuig type Bromfiets of Snorfiet -> Brom & Snor
        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    $rb['vehicleType']->equalTo(Vehicle::TYPE_SCOOTER),
                    $rb->logicalAnd(
                        $rb['engineCapacity']->greaterThanOrEqualTo(1),
                        $rb['engineCapacity']->lessThan(50)
                    )
                ),
                self::GROUP_SCOOTER_ID
            )
        );

        // 3. Voertuig type Trike of Quad -> Jong & Fris:
        $this->addRule(
            $rb->create(
                $rb['bodyType']->equalTo(BodyType::TRIKE),
                self::GROUP_YOUNG_ID
            )
        );

        // 4. Voertuig type Klassier en merk BMW, Ducati, Benelli, Moto Guzzi, MZ of EML -> Overig
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    new LogicalOr(
                        [
                            $rb['modelName']->stringContainsInsensitive('BMW'),
                            $rb['modelName']->stringContainsInsensitive('DUCATI'),
                            $rb['modelName']->stringContainsInsensitive('BENELLI'),
                            $rb['modelName']->stringContainsInsensitive('MOTO GUZZI'),
                            $rb['modelName']->stringContainsInsensitive('MZ'),
                            $rb['modelName']->stringContainsInsensitive('EML'),
                        ]
                    ),
                    $rb['bodyType']->equalTo(BodyType::CLASSIC)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // 5. Heeft het voertuig een zijspan -> Overige:
        $this->addRule(
            $rb->create(
                $rb['bodyType']->equalTo(BodyType::SIDE_CAR),
                self::GROUP_OTHER_ID
            )
        );

        // 6. Is het voertuig van het merk Honda en bevat het model de tekst Goldwing of Gold wing
        // en is het voertuig jonger dan 15 jaar => Jong & Fris
        // 7. Is het voertuig van het merk Harley Davison => Jong en Fris
        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    new LogicalAnd(
                        [
                            $rb['makeName']->equalTo('HONDA'),
                            $rb['modelName']->stringContainsInsensitive('GOLDWING'),
                            $rb['yearOfManufacture']->greaterThan($currentYear - 15)
                        ]
                    ),
                    $rb['makeName']->stringContainsInsensitive('HARLEY')
                ),
                self::GROUP_YOUNG_ID
            )
        );

        // 8. Bevat het model de tekst Valkyrie, FJR, Wildstar of Midnight star en is het voertuig
        // ouder dan 9 jaar -> Overige
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    new LogicalOr(
                        [
                            $rb['modelName']->stringContainsInsensitive('VALKYRIE'),
                            $rb['modelName']->stringContainsInsensitive('FJR'),
                            $rb['modelName']->stringContainsInsensitive('WILDSTAR'),
                            $rb['modelName']->stringContainsInsensitive('MIDNIGHT STAR'),
                        ]
                    ),
                    $rb['yearOfManufacture']->lessThan($currentYear - 9)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // 9. Bevat het model de tekst Goldwing of Gold wing en is het voertuig ouder dan 15 jaar -> Overig
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb->logicalOr(
                        $rb['modelName']->stringContainsInsensitive('GOLDWING'),
                        $rb['modelName']->stringContainsInsensitive('GOLD WING')
                    ),
                    $rb['yearOfManufacture']->lessThan($currentYear - 15)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // 10. Is het voertuig van het merk BMW en heeft het minder dan 100.000km en is het een voertuig
        // tussen de 10 en 20 jaar oud -> Overige
        $this->addRule(
            $rb->create(
                new LogicalAnd(
                    [
                        $rb['makeName']->equalTo('BMW'),
                        $rb['mileage']->lessThan(100000),
                        $rb['yearOfManufacture']->greaterThan($currentYear - 20),
                        $rb['yearOfManufacture']->lessThan($currentYear - 10)
                    ]
                ),
                self::GROUP_OTHER_ID
            )
        );

        // 11. Is het voertuig ouder dan 35 jaar -> Overig
        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->lessThan($currentYear - 35),
                self::GROUP_OTHER_ID
            )
        );

        // 12. Is het voertuig jonger dan 9 jaar -> Jong & Fris
        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->greaterThan($currentYear - 9),
                self::GROUP_YOUNG_ID
            )
        );

        // 13. Is het voertuig jonger dan 15 jaar en heeft het minder dan 70.000km -> Overige
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb['yearOfManufacture']->greaterThan($currentYear - 15),
                    $rb['mileage']->lessThan(70000)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // 14. Is het voertuig jonger dan 15 jaar -> Budget
        $this->addRule(
            $rb->create(
                $rb['yearOfManufacture']->greaterThan($currentYear - 15),
                self::GROUP_BUDGET_ID
            )
        );

        // 15. Is het voertuig jonger dan 20 jaar en heeft het minder dan 40.000km -> Overige
        $this->addRule(
            $rb->create(
                $rb->logicalAnd(
                    $rb['yearOfManufacture']->greaterThan($currentYear - 20),
                    $rb['mileage']->lessThan(40000)
                ),
                self::GROUP_OTHER_ID
            )
        );

        // 16. Is het voertuig jonger dan 20 jaar en heeft het minder dan 70.000km -> Budget
        // 17. Is het voertuig jonger dan 25 jaar en heeft het minder dan 60.000km -> Budget
        $this->addRule(
            $rb->create(
                $rb->logicalOr(
                    $rb->logicalAnd(
                        $rb['yearOfManufacture']->greaterThan($currentYear - 20),
                        $rb['mileage']->lessThan(70000)
                    ),
                    $rb->logicalAnd(
                        $rb['yearOfManufacture']->greaterThan($currentYear - 25),
                        $rb['mileage']->lessThan(60000)
                    )
                ),
                self::GROUP_BUDGET_ID
            )
        );

        // Catch all
        $this->addRule(
            $rb->create(
                new CatchAll(),
                self::GROUP_DEMOLITION_ID
            )
        );
    }
}
