<?php

namespace DealerDirect\VehicleClassification\Rules\Cars;

use DealerDirect\VehicleClassification\Models\CarOption;
use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Utils\ConditionUtils;
use Ruler\Context;
use Ruler\Operator;
use Ruler\Rule;

class GroupExportRules implements GroupRulesInterface
{
    /**
     * @var Rule
     */
    private $groupRule;

    public function __construct()
    {
        $currentYear = (int)date('Y');

        $this->groupRule = new Rule(
            new Operator\LogicalOr(array(
                // Export > 190000, 5 doors
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_CAR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::equalTo('drivable', true),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 15),
                    ConditionUtils::greaterThan('mileage', 190000),
                    ConditionUtils::equalTo('numberOfDoors', 5)
                    // Commented out for possible future use:
                    // Condition::hasValues('options', array(CarOption::AIR_CONDITIONING))

                )),
                // Export > 220000, 3 doors
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_CAR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::equalTo('drivable', true),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 11),
                    ConditionUtils::greaterThan('mileage', 220000),
                    ConditionUtils::equalTo('numberOfDoors', 3)
                    // Commented out for possible future use:
                    // Condition::hasValues('options', array(CarOption::AIR_CONDITIONING))
                )),
                // Engine/drivable damage, WITH remaining BPM
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_CAR),
                    new Operator\LogicalOr(array(
                        ConditionUtils::equalTo('damageFree', false),
                        ConditionUtils::equalTo('drivable', false)
                    )),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 12),
                    ConditionUtils::greaterThan('bpm', 0)
                )),
            ))
        );
    }

    /**
     * @param Context $context
     * @return bool
     */
    public function evaluate(Context $context)
    {
        return $this->groupRule->evaluate($context);
    }

    /**
     * Returns VCL internal group ID
     *
     * @return int
     */
    public function getGroupId()
    {
        return GroupRulesInterface::GROUP_CARS_EXPORT_ID;
    }
}
