<?php

namespace DealerDirect\VehicleClassification\Rules\Cars;

use DealerDirect\VehicleClassification\Models\Vehicle;
use DealerDirect\VehicleClassification\Rules\GroupRulesInterface;
use DealerDirect\VehicleClassification\Utils\ConditionUtils;
use Ruler\Context;
use Ruler\Operator;
use Ruler\Rule;

class GroupHighRules implements GroupRulesInterface
{
    /**
     * @var Rule
     */
    private $groupRule;

    /**
     * GroupHighRules constructor.
     */
    public function __construct()
    {
        $currentYear = (int)date('Y');

        $this->groupRule = new Rule(
            new Operator\LogicalOr(array(
                // High 110.000 - 190.000
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_CAR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::equalTo('drivable', true),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 12),
                    ConditionUtils::lessThanOrEqualTo('mileage', 190000),
                    ConditionUtils::greaterThanOrEqualTo('mileage', 110000)
                )),
                // High < 110.000
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_CAR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::equalTo('drivable', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 9),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 12),
                    ConditionUtils::lessThan('mileage', 110000)
                )),
                // High > 190.000, no 5 doors / airco
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_CAR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::equalTo('drivable', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 9),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 12),
                    ConditionUtils::greaterThanOrEqualTo('mileage', 190000),
                    ConditionUtils::notEqualTo('numberOfDoors', 5)
                    // NOTE: If we should filter by airco we miss many classifications
                )),
                // askingprice > 5000
                new Operator\LogicalAnd(array(
                    ConditionUtils::equalTo('vehicleType', Vehicle::TYPE_CAR),
                    ConditionUtils::equalTo('damageFree', true),
                    ConditionUtils::equalTo('drivable', true),
                    ConditionUtils::lessThanOrEqualTo('yearOfManufacture', $currentYear - 13),
                    ConditionUtils::greaterThanOrEqualTo('yearOfManufacture', $currentYear - 14),
                    ConditionUtils::notEqualTo('numberOfDoors', 5),
                    ConditionUtils::greaterThan('askingPrice', 5000)
                ))
            ))
        );
    }

    /**
     * @param Context $context
     * @return bool
     */
    public function evaluate(Context $context)
    {
        return $this->groupRule->evaluate($context);
    }

    /**
     * Returns VCL internal group ID
     *
     * @return int
     */
    public function getGroupId()
    {
        return GroupRulesInterface::GROUP_CARS_HIGH_ID;
    }

}
