<?php

namespace DealerDirect\VehicleClassification\Rules\Cars;

use DealerDirect\VehicleClassification\Models\CarOption;
use DealerDirect\VehicleClassification\Models\FuelUtils;
use DealerDirect\VehicleClassification\Models\Mappers\VehicleModelMapper;
use DealerDirect\VehicleClassification\Models\Vehicle;
use Ruler\Context;

class GroupExportRulesTest extends \PHPUnit_Framework_TestCase
{
    /**
     * Test Export group ruleset
     *
     * @param array $vehicleData The vehicle as array
     *
     * @dataProvider provideCorrectDataGroupExport
     */
    public function testGroupExportCorrectData(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        // Check rules
        $groupExportRules = new GroupExportRules();
        $context = new Context($vehicleData);

        $this->assertTrue($groupExportRules->evaluate($context));
    }

    /**
     * Test young group ruleset
     *
     * @param array $vehicleData The vehicle as array
     *
     * @dataProvider provideWrongDataGroupExport
     */
    public function testGroupExportWrongData(array $vehicleData)
    {
        // Map vehicle
        $mapper = new VehicleModelMapper();
        $vehicle = $mapper->toVehicle($vehicleData);
        $this->assertInstanceOf(Vehicle::class, $vehicle);

        // Check rules
        $groupExportRules = new GroupExportRules();
        $context = new Context($vehicleData);

        $this->assertFalse($groupExportRules->evaluate($context));
    }

    /**
     * @return array<array<array>>
     */
    public static function provideCorrectDataGroupExport()
    {
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'mileage' => 204000,
                'makeName' => 'AUDI',
                'modelName' => 'A3',
                'yearOfManufacture' => 2004,
                'fuelType' => '1',
                'bpm' => '409',
                'askingPrice' => 30000,
                'numberOfDoors' => 5,
                'options' => array(
                    CarOption::AIR_CONDITIONING,
                    CarOption::ABS,
                    CarOption::POWER_STEERING,
                    CarOption::ALLOY_WHEELS,
                    CarOption::CRUISE_CONTROL
                )
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '1',
                'mileage' => 230000,
                'makeName' => 'SKODA',
                'modelName' => 'SUPERB',
                'yearOfManufacture' => 2008,
                'fuelType' => '1',
                'bpm' => '8284',
                'askingPrice' => 17000,
                'numberOfDoors' => 3
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '1',
                'mileage' => 232000,
                'makeName' => 'SKODA',
                'modelName' => 'SUPERB - 2 DOOR COUPE',
                'yearOfManufacture' => 2008,
                'fuelType' => '1',
                'bpm' => '8284',
                'askingPrice' => 17000,
                'numberOfDoors' => 2
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '1',
                'mileage' => 232000,
                'makeName' => 'SKODA',
                'modelName' => 'SUPERB - 4 DOOR SEDAN',
                'yearOfManufacture' => 2008,
                'fuelType' => '1',
                'bpm' => '8284',
                'askingPrice' => 17000,
                'numberOfDoors' => 4
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '1',
                'mileage' => 232000,
                'makeName' => 'SKODA',
                'modelName' => 'SUPERB - 5 DOOR STATION',
                'yearOfManufacture' => 2008,
                'fuelType' => '1',
                'bpm' => '8284',
                'askingPrice' => 17000,
                'numberOfDoors' => 5
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => 0,
                'drivable' => 1,
                'mileage' => '73076',
                'makeName' => 'CHEVROLET',
                'modelName' => 'MATIZ',
                'yearOfManufacture' => 2006,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => 1968,
                'askingPrice' => 1850
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => false,
                'mileage' => '113000',
                'makeName' => 'SEAT',
                'modelName' => 'IBIZA',
                'yearOfManufacture' => 2005,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => 3024,
                'askingPrice' => 0
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => true,
                'drivable' => true,
                'mileage' => 204000,
                'makeName' => 'AUDI',
                'modelName' => 'A3 - 4 DOOR SEDAN',
                'yearOfManufacture' => 2005,
                'fuelType' => '1',
                'bpm' => '409',
                'askingPrice' => 29000,
                'numberOfDoors' => 4,
                'options' => array(
                    CarOption::AIR_CONDITIONING,
                    CarOption::ABS,
                    CarOption::POWER_STEERING,
                    CarOption::ALLOY_WHEELS,
                    CarOption::CRUISE_CONTROL
                )
            ))
        );
    }

    /**
     * @return array<array<array>>
     */
    public static function provideWrongDataGroupExport()
    {
        return array(
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '1',
                'mileage' => 155000,
                'makeName' => 'AUDI',
                'modelName' => 'A3',
                'yearOfManufacture' => 2015,
                'fuelType' => '1',
                'bpm' => '409',
                'askingPrice' => 30000,
                'numberOfDoors' => 5
            )),
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '1',
                'mileage' => 190000,
                'makeName' => 'SKODA',
                'modelName' => 'SUPERB',
                'yearOfManufacture' => 2012,
                'fuelType' => '1',
                'bpm' => '8284',
                'askingPrice' => 17000,
                'numberOfDoors' => 3
            )),
            // Case damaged, NO bpm
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => 0,
                'drivable' => 1,
                'mileage' => '73076',
                'makeName' => 'CHEVROLET',
                'modelName' => 'MATIZ',
                'yearOfManufacture' => 2006,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => 0,
                'askingPrice' => 1850
            )),
            // Case not drivable + bpm, but not within correct year
            array(array(
                'vehicleType' => Vehicle::TYPE_CAR,
                'damageFree' => '1',
                'drivable' => '0',
                'mileage' => '113000',
                'makeName' => 'SEAT',
                'modelName' => 'IBIZA',
                'yearOfManufacture' => 2002,
                'fuelType' => FuelUtils::GASOLINE,
                'bpm' => 3024,
                'askingPrice' => 0
            ))
        );
    }
}
